import { Head } from '@inertiajs/react';
import DashboardLayout from '@/layouts/dashboard-layout';
import { PageProps } from '@/types';
import {
    FileText,
    Globe,
    Heart,
    PiggyBank,
    Shield,
    Bike,
    Wallet,
    LayoutGrid,
    Film,
    QrCode,
    BanknoteArrowUp,
    BanknoteArrowDown,
    Send
} from 'lucide-react';
import { cn } from '@/lib/utils';
import News from '@/components/user/news';
import Topup from '@/components/user/transactions/Topup';
import Withdrawal from '@/components/user/transactions/Withdrawal';
import UpcomingFeature from '@/components/upcoming-feature';
import { useEffect, useState } from 'react';
import { apiClient } from '@/lib/api';
import AccountBank from '@/components/user/profile/account-bank';

export default function Dashboard({ auth }: PageProps) {
    const services = [
        { icon: <Shield className="h-5 w-5 text-blue-400" />, label: 'Secure2u', background: 'bg-blue-100', tag: '' },
        {
            icon: <Bike className="h-5 w-5 text-green-400" />,
            label: 'Motorcycle Insurance',
            background: 'bg-green-100',
            tag: '',
            isFeatured: false
        },
        {
            icon: <FileText className="h-5 w-5 text-purple-400" />,
            label: 'Pay Bills',
            background: 'bg-purple-100',
            tag: '',
            isFeatured: false
        },
        {
            icon: <Send className="h-5 w-5 text-orange-400" />,
            label: 'Transfer',
            background: 'bg-orange-100',
            tag: '',
            isFeatured: false
        },
        {
            icon: <Wallet className="h-5 w-5 text-red-400" />,
            label: 'Get Cashback',
            background: 'bg-red-100',
            tag: '',
            isFeatured: false
        },
        {
            icon: <PiggyBank className="h-5 w-5 text-cyan-400" />,
            label: 'Tabung',
            background: 'bg-cyan-100',
            tag: 'PROMO',
            isFeatured: false
        },
        {
            icon: <LayoutGrid className="h-5 w-5 text-yellow-400" />,
            label: 'Split Bill',
            background: 'bg-yellow-100',
            tag: '',
            isFeatured: false
        },
        {
            icon: <Film className="h-5 w-5 text-pink-400" />,
            label: 'Movies & Leisure',
            background: 'bg-pink-100',
            tag: '',
            isFeatured: false
        },
        {
            icon: <Heart className="h-5 w-5 text-green-400" />,
            label: 'Medical Insurance',
            background: 'bg-green-100',
            tag: 'NEW',
            isFeatured: true
        },
        {
            icon: <PiggyBank className="h-5 w-5 text-blue-400" />,
            label: 'Goal Based Investment',
            background: 'bg-blue-100',
            tag: 'NEW',
            isFeatured: true
        },
        {
            icon: <Wallet className="h-5 w-5 text-orange-400" />,
            label: 'Gold Investment',
            background: 'bg-orange-100',
            tag: 'NEW',
            isFeatured: true
        },
        {
            icon: <Globe className="h-5 w-5 text-purple-400" />,
            label: 'Global Access',
            background: 'bg-purple-100',
            tag: 'NEW',
            isFeatured: true
        }
    ];
    const [bankAccounts, setBankAccounts] = useState([]);

    useEffect(() => {
        fetchBankAccounts();
    }, []);

    const fetchBankAccounts = async () => {
        try {
            const response = await apiClient.get('/bank-accounts');
            if (response.ok) {
                const data = await response.json();
                setBankAccounts(data.bank_accounts || []);
            }
        } catch (error) {
            console.error('Error fetching bank accounts:', error);
        }
    };

    return (
        <DashboardLayout auth={auth}>
            <Head title="Dashboard" />

            <div className="m-4 space-y-3">
                {/* Quick Action */}
                <div className="bg-white rounded-lg px-4">
                    <div className="grid grid-cols-4 gap-2">
                        <Topup trigger={
                            <button
                                className="cursor-pointer flex flex-col items-center gap-1 p-2 hover:bg-primary/20"
                            >
                                <div className="w-10 h-10 bg-gray-100 rounded-lg flex items-center justify-center">
                                    <div className="text-gray-600">
                                        <BanknoteArrowUp className="h-5 w-5" />
                                    </div>
                                </div>
                                <span className="text-gray-700 text-xs text-center leading-tight">Tambah Baki</span>
                            </button>
                        } />
                        {
                            bankAccounts.length > 0 && (
                                <Withdrawal trigger={
                                    <button
                                        className="cursor-pointer flex flex-col items-center gap-1 p-2 hover:bg-primary/20"
                                    >
                                        <div className="w-10 h-10 bg-gray-100 rounded-lg flex items-center justify-center">
                                            <div className="text-gray-600">
                                                <BanknoteArrowDown className="h-5 w-5" />
                                            </div>
                                        </div>
                                        <span
                                            className="text-gray-700 text-xs text-center leading-tight">Withdrawal Baki</span>
                                    </button>
                                } />
                            )
                        }
                        {
                            bankAccounts.length === 0 && (
                                <AccountBank
                                    onSuccess={fetchBankAccounts}
                                    trigger={
                                        <button
                                            className="cursor-pointer flex flex-col items-center gap-1 p-2 hover:bg-primary/20"
                                        >
                                            <div
                                                className="w-10 h-10 bg-gray-100 rounded-lg flex items-center justify-center">
                                                <div className="text-gray-600">
                                                    <BanknoteArrowDown className="h-5 w-5" />
                                                </div>
                                            </div>
                                            <span className="text-gray-700 text-xs text-center leading-tight">Withdrawal Baki</span>
                                        </button>
                                    } />
                            )
                        }
                        <UpcomingFeature
                            trigger={
                                <button
                                    className="cursor-pointer flex flex-col items-center gap-1 p-2 hover:bg-primary/20"
                                >
                                    <div className="w-10 h-10 bg-gray-100 rounded-lg flex items-center justify-center">
                                        <div className="text-gray-600">
                                            <Send className="h-5 w-5" />
                                        </div>
                                    </div>
                                    <span
                                        className="text-gray-700 text-xs text-center leading-tight">Transfer Bank</span>
                                </button>
                            }
                        />
                        <UpcomingFeature
                            trigger={
                                <button
                                    className="cursor-pointer flex flex-col items-center gap-1 p-2 hover:bg-primary/20"
                                >
                                    <div className="w-10 h-10 bg-gray-100 rounded-lg flex items-center justify-center">
                                        <div className="text-gray-600">
                                            <QrCode className="h-5 w-5" />
                                        </div>
                                    </div>
                                    <span className="text-gray-700 text-xs text-center leading-tight">Tambah Baki</span>
                                </button>
                            }
                        />
                    </div>
                </div>
                {/* Services Grid */}
                <div className="bg-white rounded-lg b-4 px-4 py-2">
                    <div className="grid grid-cols-4 gap-2">
                        {
                            services.filter((service) => !service.isFeatured).map((service, idx) => (
                                <UpcomingFeature
                                    key={idx}
                                    trigger={
                                        <button
                                            className="cursor-pointer flex flex-col items-center gap-2 p-2 relative hover:bg-primary/20 rounded-md">
                                            {service.tag && <span
                                                className="absolute top-0 left-0 text-[.6rem] bg-blue-500 text-white rounded-md px-2 py-0.5">{service.tag}</span>}
                                            <div className={cn(
                                                'w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center',
                                                service.background
                                            )}>
                                                {service.icon}
                                            </div>
                                            <span
                                                className="text-gray-700 text-xs text-center leading-tight">{service.label}</span>
                                        </button>
                                    }
                                />
                            ))
                        }
                    </div>
                </div>
                {/* Featured Grid */}
                <div className="bg-white rounded-lg b-4 px-4 py-2 space-y-2">
                    <h3 className="font-bold text-black">Featured</h3>
                    <div className="grid grid-cols-4 gap-2">
                        {
                            services.filter((service) => service.isFeatured).map((service, idx) => (
                                <UpcomingFeature
                                    key={idx}
                                    trigger={
                                        <button
                                            className="cursor-pointer flex flex-col items-center gap-2 p-2 relative hover:bg-primary/20 rounded-md">
                                            {service.tag && <span
                                                className="absolute top-0 left-0 text-[.6rem] bg-blue-500 text-white rounded-md px-2 py-0.5">{service.tag}</span>}
                                            <div className={cn(
                                                'w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center',
                                                service.background
                                            )}>
                                                {service.icon}
                                            </div>
                                            <span
                                                className="text-gray-700 text-xs text-center leading-tight">{service.label}</span>
                                        </button>
                                    }
                                />
                            ))
                        }
                    </div>
                </div>
                {/* News Section */}
                <News />
            </div>
        </DashboardLayout>
    );
}
