import { Head, Link, router, usePage } from '@inertiajs/react';
import { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import {
    Plus,
    Search,
    Filter,
    Edit,
    Eye,
    Trash2,
    MoreHorizontal, Pencil, Wallet
} from 'lucide-react';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuTrigger
} from '@/components/ui/dropdown-menu';
import AdminLayout from '@/layouts/admin-layout';
import Swal from 'sweetalert2';
import CreateUser from '@/pages/admin/users/create';
import BalanceManagement from '@/components/admin/balance-management';
import Pagination from '@/components/ui/pagination';
import { PageProps, PaginationProps, User } from '@/types';
import { formatAmount } from '@/lib/utils';
import EmptyData from '@/components/empty-data';

interface UsersIndexProps {
    users: PaginationProps<User>;
    filters: {
        search?: string;
        role?: string;
    };
}

export default function UsersIndex({ users, filters }: UsersIndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [role, setRole] = useState(filters.role || '');
    const [isOpen, setIsOpen] = useState(false);
    const [editingUser, setEditingUser] = useState<User | null>(null);
    const [balanceManagementOpen, setBalanceManagementOpen] = useState(false);
    const [selectedUser, setSelectedUser] = useState<User | null>(null);
    const { auth } = usePage<PageProps>().props;

    const handleSearch = () => {
        router.get(route('admin.users.index'), { search, role }, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const handleReset = () => {
        setSearch('');
        setRole('');
        router.get(route('admin.users.index'), {}, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const handleDelete = (userId: number, userName: string) => {
        Swal.fire({
            title: 'Adakah anda pasti?',
            text: `Anda akan memadamkan pengguna "${userName}". Tindakan ini tidak dapat dibatalkan!`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Ya, Padam!',
            cancelButtonText: 'Batal'
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('admin.users.destroy', userId), {
                    onSuccess: () => {
                        Swal.fire(
                            'Dipadam!',
                            'Pengguna telah berjaya dipadamkan.',
                            'success'
                        );
                    },
                    onError: () => {
                        Swal.fire(
                            'Ralat!',
                            'Gagal memadamkan pengguna.',
                            'error'
                        );
                    }
                });
            }
        });
    };

    const handleCreate = () => {
        setIsOpen(true);
    };

    const handleEdit = (user: User) => {
        setEditingUser(user);
        setIsOpen(true);
    };

    const handleClose = () => {
        setIsOpen(false);
        setEditingUser(null);
    };

    const handleBalanceManagement = (user: User) => {
        setSelectedUser(user);
        setBalanceManagementOpen(true);
    };

    const handleBalanceManagementClose = () => {
        setBalanceManagementOpen(false);
        setSelectedUser(null);
    };

    const handleBalanceUpdated = (newBalance: number) => {
        if (selectedUser) {
            // Update the user's balance in the local state
            // Update the users object
            users.data = users.data.map(user =>
                user.id === selectedUser.id
                    ? { ...user, balance: newBalance }
                    : user
            );
        }
    };

    const handleToggleVerification = (user: User) => {
        Swal.fire({
            title: 'Adakah anda pasti?',
            text: `Anda akan ${user.is_verified ? 'membatalkan pengesahan' : 'mengesahkan'} pengguna "${user.name}".`,
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            confirmButtonText: 'Ya, Teruskan!',
            cancelButtonText: 'Batal'
        }).then((result) => {
            if (result.isConfirmed) {
                router.patch(route('admin.users.toggle-verification', user.id), {}, {
                    onSuccess: () => {
                        Swal.fire(
                            'Berjaya!',
                            `Status pengesahan pengguna telah ${user.is_verified ? 'dibatalkan' : 'disahkan'}.`,
                            'success'
                        );
                    },
                    onError: () => {
                        Swal.fire(
                            'Ralat!',
                            'Gagal mengubah status pengesahan pengguna.',
                            'error'
                        );
                    }
                });
            }
        });
    };

    const formatDate = (dateString: string) => {
        return new Date(dateString).toLocaleDateString('ms-MY', {
            day: '2-digit',
            month: 'short',
            year: 'numeric'
        });
    };

    return (
        <AdminLayout>
            <Head title="Pengurusan Pengguna" />

            <div className="p-6 space-y-6">
                {/* Header */}
                <div className="flex justify-between items-center">
                    <div>
                        <h1 className="text-3xl font-bold text-gray-900">Pengurusan Pengguna</h1>
                        <p className="text-gray-600">Urus semua pengguna dalam sistem</p>
                    </div>
                    <Button onClick={handleCreate}>
                        <Plus className="w-4 h-4 mr-2" />
                        Tambah Pengguna
                    </Button>
                </div>

                {/* Filters */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center">
                            <Filter className="w-4 h-4 mr-2" />
                            Penapis
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="flex flex-col sm:flex-row gap-4">
                            <div className="flex-1">
                                <Input
                                    placeholder="Cari nama, email, atau nombor telefon..."
                                    value={search}
                                    onChange={(e) => setSearch(e.target.value)}
                                    onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
                                />
                            </div>
                            <div className="w-full sm:w-48">
                                <Select value={role} onValueChange={setRole}>
                                    <SelectTrigger>
                                        <SelectValue placeholder="Semua Peranan" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">Semua Peranan</SelectItem>
                                        <SelectItem value="admin">Admin</SelectItem>
                                        <SelectItem value="user">Pengguna</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="flex gap-2">
                                <Button onClick={handleSearch} variant="outline">
                                    <Search className="w-4 h-4 mr-2" />
                                    Cari
                                </Button>
                                <Button onClick={handleReset} variant="outline">
                                    Reset
                                </Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Users Table */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center justify-between">
                            <span>Senarai Pengguna ({users.total})</span>
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                <tr className="border-b">
                                    <th className="text-left py-3 px-4 font-medium">Nama</th>
                                    <th className="text-left py-3 px-4 font-medium">Email</th>
                                    <th className="text-left py-3 px-4 font-medium">Telefon</th>
                                    <th className="text-left py-3 px-4 font-medium">Peranan</th>
                                    <th className="text-left py-3 px-4 font-medium">Baki</th>
                                    <th className="text-left py-3 px-4 font-medium">Pengesahan</th>
                                    <th className="text-left py-3 px-4 font-medium">Dicipta</th>
                                    <th className="text-left py-3 px-4 font-medium">Tindakan</th>
                                </tr>
                                </thead>
                                <tbody>
                                {users.data.map((user) => (
                                    <tr key={user.id} className="border-b hover:bg-gray-50">
                                        <td className="py-3 px-4">
                                            <div>
                                                <p className="font-medium">{user.name}</p>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <p className="text-sm text-gray-600">{user.email}</p>
                                        </td>
                                        <td className="py-3 px-4">
                                            <p className="text-sm text-gray-600">{user.phone_number}</p>
                                        </td>
                                        <td className="py-3 px-4">
                                            <Badge variant={user.role === 'admin' ? 'destructive' : 'secondary'}>
                                                {user.role === 'admin' ? 'Admin' : 'Pengguna'}
                                            </Badge>
                                        </td>
                                        <td className="py-3 px-4">
                                            {user.role === 'user' ? (
                                                <Button
                                                    variant="ghost"
                                                    size="sm"
                                                    onClick={() => handleBalanceManagement(user)}
                                                    className="p-0 hover:bg-green-100 cursor-pointer px-4"
                                                >
                                                    <div className="flex items-center gap-1">
                                                        <span className="font-medium text-green-600">
                                                            {formatAmount(user.balance ?? 0)}
                                                        </span>
                                                        <Wallet className="w-3 h-3 text-green-600" />
                                                    </div>
                                                </Button>
                                            ) : (
                                                <span className="text-gray-400">-</span>
                                            )}
                                        </td>
                                        <td className="py-3 px-4">
                                            {
                                                user.role === 'user' && (
                                                    <div className="flex items-center gap-2">
                                                        <Badge
                                                            className="cursor-pointer"
                                                            onClick={() => handleToggleVerification(user)}
                                                            variant={user.is_verified ? 'default' : 'secondary'}>
                                                                <span>
                                                                {user.is_verified ? 'Disahkan' : 'Belum Disahkan'}
                                                                </span>
                                                            <Pencil />
                                                        </Badge>
                                                    </div>
                                                )
                                            }
                                        </td>
                                        <td className="py-3 px-4">
                                            <p className="text-sm text-gray-600">{formatDate(user.created_at)}</p>
                                        </td>
                                        <td className="py-3 px-4">
                                            <DropdownMenu>
                                                <DropdownMenuTrigger asChild>
                                                    <Button variant="ghost" size="sm">
                                                        <MoreHorizontal className="w-4 h-4" />
                                                    </Button>
                                                </DropdownMenuTrigger>
                                                <DropdownMenuContent align="end">
                                                    <DropdownMenuItem asChild>
                                                        <Link href={route('admin.users.show', user.id)}>
                                                            <Eye className="w-4 h-4 mr-2" />
                                                            Lihat
                                                        </Link>
                                                    </DropdownMenuItem>
                                                    <DropdownMenuItem onClick={() => handleEdit(user)}>
                                                        <Edit className="w-4 h-4 mr-2" />
                                                        Edit
                                                    </DropdownMenuItem>
                                                    {auth.user.id !== user.id && <DropdownMenuItem
                                                        onClick={() => handleDelete(user.id, user.name)}
                                                        className="text-red-600"
                                                    >
                                                        <Trash2 className="w-4 h-4 mr-2" />
                                                        Padam
                                                    </DropdownMenuItem>}
                                                </DropdownMenuContent>
                                            </DropdownMenu>
                                        </td>
                                    </tr>
                                ))}
                                </tbody>
                            </table>
                            {users.data.length === 0 && <EmptyData theme="light" />}
                        </div>

                        {/* Pagination */}
                        <Pagination
                            current_page={users.current_page}
                            last_page={users.last_page}
                            total={users.total}
                            per_page={users.per_page}
                            links={users.links}
                        />
                    </CardContent>
                </Card>
            </div>

            {/* Create User Sheet */}
            <CreateUser
                isOpen={isOpen}
                onClose={handleClose}
                defaultValues={editingUser}
            />

            {/* Balance Management Modal */}
            {selectedUser && (
                <BalanceManagement
                    user={selectedUser}
                    isOpen={balanceManagementOpen}
                    onClose={handleBalanceManagementClose}
                    onBalanceUpdated={handleBalanceUpdated}
                />
            )}
        </AdminLayout>
    );
}
