import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from "@/components/ui/sheet";
import { router } from "@inertiajs/react";
import { useEffect, useState } from "react";
import Swal from "sweetalert2";

type Props = {
  isOpen: boolean;
  onClose: () => void;
  defaultValues: DefaultValues | null;
};

type DefaultValues = {
  id?: number | null;
  name: string;
  email: string;
  phone_number: string;
  role: string;
};

type FormValues = {
  id: number | null;
  name: string;
  email: string;
  phone_number: string;
  role: string;
  password: string;
  password_confirmation: string;
};

const InitValues: FormValues = {
  id: null,
  name: "",
  email: "",
  phone_number: "",
  role: "user",
  password: "",
  password_confirmation: "",
};

const CreateUser = ({ defaultValues = InitValues, isOpen, onClose }: Props) => {
  const [formData, setFormData] = useState<FormValues>(InitValues);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [isSubmitting, setIsSubmitting] = useState(false);

  const handleSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setIsSubmitting(true);
    setErrors({});

    // Only validate match if either password field is filled
    if (
      (formData.password || formData.password_confirmation) &&
      formData.password !== formData.password_confirmation
    ) {
      setErrors({ password_confirmation: "Kata laluan tidak sepadan." });
      setIsSubmitting(false);
      return;
    }

    const isEdit = Boolean(formData.id);

    // If editing and both password fields are empty, strip them from payload
    let payload: Record<string, any> = { ...formData };
    if (isEdit && !formData.password && !formData.password_confirmation) {
      const { password, password_confirmation, ...rest } = formData;
      payload = rest;
    }

    if (isEdit) {
      router.put(route("admin.users.update", formData.id as number), payload, {
        onSuccess: () => {
          handleClose();
          Swal.fire("Berjaya!", "Pengguna telah berjaya dikemas kini.", "success");
        },
        onError: (errs: any) => {
          setErrors(errs);
          setIsSubmitting(false);
        },
      });
    } else {
      router.post(route("admin.users.store"), payload, {
        onSuccess: () => {
          handleClose();
          Swal.fire("Berjaya!", "Pengguna telah berjaya dicipta.", "success");
        },
        onError: (errs: any) => {
          setErrors(errs);
          setIsSubmitting(false);
        },
      });
    }
  };

  const handleClose = () => {
    setFormData(InitValues);
    setErrors({});
    setIsSubmitting(false);
    onClose();
  };

  useEffect(() => {
    if (isOpen && defaultValues) {
      setFormData({
        ...InitValues,
        ...defaultValues,
        id: defaultValues.id ?? null,
        password: "",
        password_confirmation: "",
      });
    } else {
      setFormData(InitValues);
    }
  }, [isOpen, defaultValues]);

  const isEdit = Boolean(formData.id);

  return (
    <Sheet open={isOpen} onOpenChange={handleClose}>
      <SheetContent side="right" className="bg-background w-[400px] sm:w-[540px]">
        <SheetHeader>
          <SheetTitle>{isEdit ? "Kemas Kini Pengguna" : "Tambah Pengguna Baru"}</SheetTitle>
          <SheetDescription>
            Isi maklumat pengguna yang akan {isEdit ? "dikemas kini" : "ditambah"} ke dalam sistem.
          </SheetDescription>
        </SheetHeader>
        <form onSubmit={handleSubmit} className="space-y-4 px-4 overflow-y-auto pb-4">
          <div className="space-y-2">
            <Label htmlFor="name">Nama</Label>
            <Input
              id="name"
              value={formData.name}
              onChange={(e) => setFormData({ ...formData, name: e.target.value })}
              className={errors.name ? "border-red-500" : ""}
              required
              error={errors.name}
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="email">Email</Label>
            <Input
              id="email"
              type="email"
              value={formData.email}
              onChange={(e) => setFormData({ ...formData, email: e.target.value })}
              className={errors.email ? "border-red-500" : ""}
              required
              error={errors.email}
              autoComplete="email"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="phone_number">Nombor Telefon</Label>
            <Input
              id="phone_number"
              value={formData.phone_number}
              onChange={(e) => setFormData({ ...formData, phone_number: e.target.value })}
              className={errors.phone_number ? "border-red-500" : ""}
              required
              error={errors.phone_number}
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="role">Peranan</Label>
            <Select
              value={formData.role}
              onValueChange={(value) => setFormData({ ...formData, role: value })}
              // If your Select supports error prop
              error={errors.role}
            >
              <SelectTrigger>
                <SelectValue placeholder="Pilih peranan" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="user">Pengguna</SelectItem>
                <SelectItem value="admin">Admin</SelectItem>
              </SelectContent>
            </Select>
          </div>

          <div className="space-y-2">
            <Label htmlFor="password">Kata Laluan</Label>
            <Input
              id="password"
              type="password"
              value={formData.password}
              onChange={(e) => setFormData({ ...formData, password: e.target.value })}
              className={errors.password ? "border-red-500" : ""}
              required={!isEdit} // only required when creating
              error={errors.password}
              autoComplete={isEdit ? "new-password" : "new-password"}
              placeholder={isEdit ? "Biarkan kosong jika tidak menukar" : undefined}
              minLength={6}
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="password_confirmation">Sahkan Kata Laluan</Label>
            <Input
              id="password_confirmation"
              type="password"
              value={formData.password_confirmation}
              onChange={(e) =>
                setFormData({ ...formData, password_confirmation: e.target.value })
              }
              className={errors.password_confirmation ? "border-red-500" : ""}
              required={!isEdit} // only required when creating
              error={errors.password_confirmation}
              autoComplete={isEdit ? "new-password" : "new-password"}
              placeholder={isEdit ? "Biarkan kosong jika tidak menukar" : undefined}
              minLength={6}
            />
          </div>

          <div className="flex gap-2 pt-4">
            <Button type="submit" disabled={isSubmitting} className="flex-1">
              {isSubmitting ? "Memproses..." : isEdit ? "Kemas Kini" : "Tambah Pengguna"}
            </Button>
            <Button
              type="button"
              variant="outline"
              onClick={handleClose}
              className="flex-1"
              disabled={isSubmitting}
            >
              Batal
            </Button>
          </div>
        </form>
      </SheetContent>
    </Sheet>
  );
};

export default CreateUser;