import { Head, router } from '@inertiajs/react';
import { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { Eye, Search, Filter, Check, X } from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import Swal from 'sweetalert2';
import { PaginationProps, Transaction } from '@/types';
import { formatAmount, formatDate } from '@/lib/utils';
import { cn } from '@/lib/utils';
import { getTransactionStatusLabel, getStatusVariant } from '@/components/user/transactions/TransactionItem';
import TransactionDetailSheet from '@/components/admin/transactions/TransactionDetailSheet';
import Pagination from '@/components/ui/pagination';
import EmptyData from '@/components/empty-data';

interface TransactionsIndexProps {
    transactions: PaginationProps<Transaction>;
    filters: {
        search?: string;
        type?: string;
        status?: string;
        date_from?: string;
        date_to?: string;
    };
    types: string[];
    statuses: string[];
}

export default function TransactionsIndex({ transactions, filters }: TransactionsIndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [type, setType] = useState(filters.type || '');
    const [status, setStatus] = useState(filters.status || '');
    const [dateFrom, setDateFrom] = useState(filters.date_from || '');
    const [dateTo, setDateTo] = useState(filters.date_to || '');
    const [selectedTransaction, setSelectedTransaction] = useState<Transaction | null>(null);
    const [isDetailSheetOpen, setIsDetailSheetOpen] = useState(false);

    const handleSearch = () => {
        router.get(route('admin.transactions.index'), {
            search,
            type,
            status,
            date_from: dateFrom,
            date_to: dateTo
        }, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const handleReset = () => {
        setSearch('');
        setType('');
        setStatus('');
        setDateFrom('');
        setDateTo('');
        router.get(route('admin.transactions.index'), {}, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const handleViewDetail = (transaction: Transaction) => {
        setSelectedTransaction(transaction);
        setIsDetailSheetOpen(true);
    };

    const handleApprove = async (transaction: Transaction) => {
        const result = await Swal.fire({
            title: 'Sahkan Transaksi?',
            text: `Adakah anda pasti mahu meluluskan transaksi ${transaction.reference}?`,
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#10b981',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'Ya, Luluskan',
            cancelButtonText: 'Batal'
        });

        if (result.isConfirmed) {
            router.patch(route('admin.transactions.update-status', transaction.id), {
                status: 'completed'
            }, {
                onSuccess: () => {
                    Swal.fire({
                        title: 'Berjaya!',
                        text: 'Transaksi telah diluluskan',
                        icon: 'success',
                        timer: 2000
                    });
                },
                onError: () => {
                    Swal.fire({
                        title: 'Ralat!',
                        text: 'Gagal meluluskan transaksi',
                        icon: 'error'
                    });
                }
            });
        }
    };

    const handleReject = async (transaction: Transaction) => {
        const result = await Swal.fire({
            title: 'Tolak Transaksi?',
            text: `Adakah anda pasti mahu menolak transaksi ${transaction.reference}?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#ef4444',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'Ya, Tolak',
            cancelButtonText: 'Batal'
        });

        if (result.isConfirmed) {
            router.patch(route('admin.transactions.update-status', transaction.id), {
                status: 'failed'
            }, {
                onSuccess: () => {
                    Swal.fire({
                        title: 'Berjaya!',
                        text: 'Transaksi telah ditolak',
                        icon: 'success',
                        timer: 2000
                    });
                },
                onError: () => {
                    Swal.fire({
                        title: 'Ralat!',
                        text: 'Gagal menolak transaksi',
                        icon: 'error'
                    });
                }
            });
        }
    };

    return (
        <AdminLayout>
            <Head title="Pengurusan Transaksi" />

            <div className="p-6 space-y-6">
                {/* Header */}
                <div className="flex justify-between items-center">
                    <div>
                        <h1 className="text-3xl font-bold text-gray-900">Pengurusan Transaksi</h1>
                        <p className="text-gray-600">Urus semua transaksi dalam sistem</p>
                    </div>
                </div>

                {/* Filters */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center">
                            <Filter className="w-4 h-4 mr-2" />
                            Penapis
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-6 gap-4">
                            <div className="lg:col-span-2">
                                <Input
                                    placeholder="Cari rujukan, penerima, atau nama pengguna..."
                                    value={search}
                                    onChange={(e) => setSearch(e.target.value)}
                                    onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
                                />
                            </div>
                            <div>
                                <Input
                                    type="date"
                                    value={dateFrom}
                                    onChange={(e) => setDateFrom(e.target.value)}
                                    placeholder="Dari tarikh"
                                />
                            </div>
                            <div>
                                <Input
                                    type="date"
                                    value={dateTo}
                                    onChange={(e) => setDateTo(e.target.value)}
                                    placeholder="Hingga tarikh"
                                />
                            </div>
                        </div>
                        <div className="flex gap-2 mt-4">
                            <Button onClick={handleSearch} variant="outline">
                                <Search className="w-4 h-4 mr-2" />
                                Cari
                            </Button>
                            <Button onClick={handleReset} variant="outline">
                                Reset
                            </Button>
                        </div>
                    </CardContent>
                </Card>

                {/* Transactions Table */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center justify-between">
                            <span>Senarai Transaksi ({transactions.total})</span>
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                <tr className="border-b">
                                    <th className="text-left py-3 px-4 font-medium">Rujukan</th>
                                    <th className="text-left py-3 px-4 font-medium">Pengguna</th>
                                    <th className="text-left py-3 px-4 font-medium">Jumlah</th>
                                    <th className="text-left py-3 px-4 font-medium">Status</th>
                                    <th className="text-left py-3 px-4 font-medium">Penerima</th>
                                    <th className="text-left py-3 px-4 font-medium">Tarikh</th>
                                    <th className="text-left py-3 px-4 font-medium">Tindakan</th>
                                </tr>
                                </thead>
                                <tbody>
                                {transactions.data.map((transaction) => (
                                    <tr key={transaction.id} className="border-b hover:bg-gray-50">
                                        <td className="py-3 px-4">
                                            <div>
                                                <p className="font-medium text-sm">{transaction.reference}</p>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div>
                                                <p className="font-medium">{transaction?.user?.name}</p>
                                                <p className="text-sm text-gray-600">{transaction?.user?.email}</p>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <p className={cn(
                                                `font-medium`,
                                                transaction.amount > 0 && 'text-green-600',
                                                transaction.amount < 0 && 'text-red-600',
                                            )}>
                                                {formatAmount(transaction.amount)}
                                            </p>
                                        </td>
                                        <td className="py-3 px-4">
                                            <Badge variant={getStatusVariant(transaction.status)}>
                                                {getTransactionStatusLabel(transaction.status)}
                                            </Badge>
                                        </td>
                                        <td className="py-3 px-4">
                                            <p className="text-sm text-gray-600">{transaction.recipient}</p>
                                        </td>
                                        <td className="py-3 px-4">
                                            <p className="text-sm text-gray-600">{formatDate(transaction.created_at)}</p>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="flex flex-row items-center space-x-1">
                                                <Button
                                                    size="icon"
                                                    onClick={() => handleViewDetail(transaction)}
                                                    title="Lihat butiran"
                                                >
                                                    <Eye className="w-4 h-4" />
                                                </Button>
                                                {
                                                    transaction.status === 'pending' && (
                                                        <>
                                                            <Button
                                                                variant="outline"
                                                                size="icon"
                                                                onClick={() => handleApprove(transaction)}
                                                                className="text-green-600 border-green-600 hover:bg-green-50"
                                                                title="Luluskan transaksi"
                                                            >
                                                                <Check className="w-4 h-4" />
                                                            </Button>
                                                            <Button
                                                                variant="outline"
                                                                size="icon"
                                                                onClick={() => handleReject(transaction)}
                                                                className="text-red-600 border-red-600 hover:bg-red-50"
                                                                title="Tolak transaksi"
                                                            >
                                                                <X className="w-4 h-4" />
                                                            </Button>
                                                        </>
                                                    )
                                                }
                                            </div>
                                        </td>
                                    </tr>
                                ))}
                                </tbody>
                            </table>
                            {transactions.data.length === 0 && <EmptyData theme="light" />}
                        </div>

                        {/* Pagination */}
                        <Pagination
                            current_page={transactions.current_page}
                            last_page={transactions.last_page}
                            total={transactions.total}
                            per_page={transactions.per_page}
                            links={transactions.links}
                        />
                    </CardContent>
                </Card>
            </div>

            {/* Transaction Detail Sheet */}
            <TransactionDetailSheet
                isOpen={isDetailSheetOpen}
                onClose={() => {
                    setIsDetailSheetOpen(false);
                    setSelectedTransaction(null);
                }}
                transaction={selectedTransaction}
            />
        </AdminLayout>
    );
}
