import { Head, Link, router, useForm } from '@inertiajs/react';
import { useState, useRef, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { ArrowLeft, Send, Paperclip, X, User, MessageCircle } from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import { Ticket } from '@/types';
import { toast } from 'react-toastify';

import { cn } from '@/lib/utils';
import EmptyData from '@/components/empty-data';

interface TicketShowProps {
    ticket: Ticket;
}

const getStatusColor = (status: string) => {
    switch (status) {
        case 'open':
            return 'text-blue-500 bg-blue-500/20';
        case 'in_progress':
            return 'text-yellow-500 bg-yellow-500/10';
        case 'resolved':
            return 'text-green-500 bg-green-500/10';
        case 'closed':
            return 'text-gray-500 bg-gray-500/10';
        default:
            return 'text-gray-500 bg-gray-500/10';
    }
};

const getStatusText = (status: string) => {
    switch (status) {
        case 'open':
            return 'Terbuka';
        case 'in_progress':
            return 'Dalam Proses';
        case 'resolved':
            return 'Selesai';
        case 'closed':
            return 'Ditutup';
        default:
            return 'Tidak Diketahui';
    }
};

const getPriorityColor = (priority: string) => {
    switch (priority) {
        case 'high':
            return 'text-red-500';
        case 'medium':
            return 'text-yellow-500';
        case 'low':
            return 'text-green-500';
        default:
            return 'text-gray-500';
    }
};

export default function TicketShow({ ticket }: TicketShowProps) {
    const [newMessage, setNewMessage] = useState('');
    const [messageFiles, setMessageFiles] = useState<File[]>([]);
    const [messagePreviews, setMessagePreviews] = useState<string[]>([]);
    const [previousRepliesCount, setPreviousRepliesCount] = useState(ticket.replies?.length || 0);
    const messageImageInputRef = useRef<HTMLInputElement>(null);

    const { data, setData, patch, processing } = useForm({
        status: ticket.status
    });

    // Previews for message images
    useEffect(() => {
        const urls = messageFiles.map((f) => URL.createObjectURL(f));
        setMessagePreviews(urls);
        return () => {
            urls.forEach((u) => URL.revokeObjectURL(u));
        };
    }, [messageFiles]);

    // Check for new user replies and show notifications
    useEffect(() => {
        const currentRepliesCount = ticket.replies?.length || 0;

        if (currentRepliesCount > previousRepliesCount) {
            // New replies were added
            const newReplies = ticket.replies?.slice(previousRepliesCount) || [];

            newReplies.forEach((reply) => {
                if (reply.user?.role !== 'admin') {
                    // This is a user reply, show notification
                    toast.info(
                        `${reply.user?.name || 'Unknown User'} replied to ticket: "${ticket.title}"`,
                        {
                            position: 'top-right',
                            autoClose: 5000,
                            hideProgressBar: false,
                            closeOnClick: true,
                            pauseOnHover: true,
                            draggable: true
                        }
                    );
                }
            });

            setPreviousRepliesCount(currentRepliesCount);
        }
    }, [ticket.replies, previousRepliesCount, ticket.title]);

    const onPickMessageFiles: React.ChangeEventHandler<HTMLInputElement> = (e) => {
        const files = e.currentTarget.files;
        if (!files || files.length === 0) return;
        const picked: File[] = [];
        for (let i = 0; i < files.length; i++) {
            const f = files.item(i);
            if (f) picked.push(f);
        }
        setMessageFiles((prev) => [...prev, ...picked]);
        e.currentTarget.value = '';
    };

    const removeMessageFile = (idx: number) => setMessageFiles((prev) => prev.filter((_, i) => i !== idx));

    const handleStatusUpdate = () => {
        patch(route('admin.tickets.update-status', ticket.id), {
            onSuccess: () => {
                // Status updated successfully
                toast.success(`Ticket status has been updated to ${data.status}`, {
                    position: 'top-right',
                    autoClose: 5000,
                    hideProgressBar: false,
                    closeOnClick: true,
                    pauseOnHover: true,
                    draggable: true
                });
            },
            onError: () => {
                // Handle error silently
                toast.error('Failed to update ticket status', {
                    position: 'top-right',
                    autoClose: 5000,
                    hideProgressBar: false,
                    closeOnClick: true,
                    pauseOnHover: true,
                    draggable: true
                });
            }
        });
    };

    const handleSendMessage = () => {
        if (!newMessage.trim() && messageFiles.length === 0) return;

        const form = new FormData();
        form.append('content', newMessage);
        messageFiles.forEach((f) => form.append('attachments[]', f));

        router.post(route('admin.tickets.reply', ticket.id), form, {
            onSuccess: () => {
                setNewMessage('');
                setMessageFiles([]);
                // Reply sent successfully
                toast.success('Your reply has been sent successfully', {
                    position: 'top-right',
                    autoClose: 5000,
                    hideProgressBar: false,
                    closeOnClick: true,
                    pauseOnHover: true,
                    draggable: true
                });
            },
            onError: () => {
                // Handle error silently
                toast.error('Failed to send reply', {
                    position: 'top-right',
                    autoClose: 5000,
                    hideProgressBar: false,
                    closeOnClick: true,
                    pauseOnHover: true,
                    draggable: true
                });
            }
        });
    };

    return (
        <AdminLayout>
            <Head title={`Tiket #${ticket.id} - ${ticket.title}`} />

            <div className="p-6 space-y-6">
                {/* Header */}
                <div className="flex items-center gap-4">
                    <Link href={route('admin.tickets.index')}>
                        <Button variant="outline" size="icon">
                            <ArrowLeft className="w-4 h-4" />
                        </Button>
                    </Link>
                    <div>
                        <h1 className="text-3xl font-bold text-gray-900">Tiket #{ticket.id}</h1>
                        <p className="text-gray-600">{ticket.title}</p>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Main Content */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Ticket Details */}
                        <Card>
                            <CardHeader>
                                <CardTitle>Maklumat Tiket</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-4">
                                    <div>
                                        <h3 className="font-medium text-gray-900">{ticket.title}</h3>
                                        <p className="text-gray-600 mt-2 whitespace-pre-wrap">{ticket.description}</p>
                                    </div>

                                    {/* Attachments */}
                                    {ticket.attachments && ticket.attachments.length > 0 && (
                                        <div>
                                            <h4 className="font-medium text-gray-900 mb-2">Lampiran</h4>
                                            <div className="grid grid-cols-2 gap-2">
                                                {ticket.attachments.map((attachment, index) => (
                                                    <div key={index} className="relative">
                                                        <img
                                                            src={`/storage/${attachment.path}`}
                                                            alt={attachment.name}
                                                            className="w-full h-32 object-cover rounded-lg"
                                                        />
                                                        <a
                                                            href={`/storage/${attachment.path}`}
                                                            target="_blank"
                                                            rel="noopener noreferrer"
                                                            className="absolute inset-0 flex items-center justify-center bg-black/50 opacity-0 hover:opacity-100 transition-opacity rounded-lg"
                                                        >
                                                            <span className="text-white text-sm">Lihat</span>
                                                        </a>
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </CardContent>
                        </Card>

                        {/* Messages */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center gap-2">
                                    <MessageCircle className="w-5 h-5" />
                                    Perbualan
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-4">
                                    {ticket.replies?.map((reply) => (
                                        <div
                                            key={reply.id}
                                            className={cn('flex gap-3', reply.user?.role === 'admin' ? 'flex-row-reverse' : 'flex-row')}
                                        >
                                            <div
                                                className={cn(
                                                    'w-8 h-8 rounded-full flex items-center justify-center flex-shrink-0',
                                                    reply.user?.role === 'admin' ? 'bg-blue-500 text-white' : 'bg-gray-200 text-gray-700'
                                                )}
                                            >
                                                <User className="w-4 h-4" />
                                            </div>
                                            <div
                                                className={cn('max-w-[80%] space-y-1', reply.user?.role === 'admin' ? 'items-end' : 'items-start')}>
                                                <div className="text-xs text-gray-500">
                                                    {reply.user?.name} • {new Date(reply.created_at).toLocaleString('ms-MY')}
                                                </div>
                                                <div
                                                    className={cn(
                                                        'p-3 rounded-lg text-sm',
                                                        reply.user?.role === 'admin'
                                                            ? 'bg-blue-500 text-white rounded-br-sm'
                                                            : 'bg-gray-100 text-gray-900 rounded-bl-sm'
                                                    )}
                                                >
                                                    {reply.content}
                                                    {reply.attachments?.length ? (
                                                        <div className="mt-2 grid grid-cols-2 gap-2">
                                                            {reply.attachments.map((att, index) => (
                                                                <a key={index} href={`/storage/${att.path}`}
                                                                   target="_blank" rel="noreferrer">
                                                                    <img src={`/storage/${att.path}`} alt={att.name}
                                                                         className="w-full h-28 object-cover rounded-md" />
                                                                </a>
                                                            ))}
                                                        </div>
                                                    ) : null}
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                    {ticket.replies?.length === 0 && <EmptyData theme='light' />}
                                </div>
                            </CardContent>
                        </Card>

                        {/* Reply Form */}
                        {ticket.status !== 'closed' && (
                            <Card>
                                <CardHeader>
                                    <CardTitle>Balas Tiket</CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        {messagePreviews.length > 0 && (
                                            <div className="grid grid-cols-4 gap-2">
                                                {messagePreviews.map((src, idx) => (
                                                    <div key={src} className="relative group">
                                                        <img src={src} alt=""
                                                             className="w-full h-20 object-cover rounded-md" />
                                                        <button
                                                            type="button"
                                                            onClick={() => removeMessageFile(idx)}
                                                            className="absolute -top-2 -right-2 rounded-full bg-black/60 p-1 opacity-0 group-hover:opacity-100 transition"
                                                            aria-label="Remove image"
                                                        >
                                                            <X className="w-3 h-3 text-white" />
                                                        </button>
                                                    </div>
                                                ))}
                                            </div>
                                        )}

                                        <div className="flex gap-2">
                                            <Input
                                                value={newMessage}
                                                onChange={(e) => setNewMessage(e.target.value)}
                                                placeholder="Tulis balasan..."
                                                className="flex-1"
                                                onKeyPress={(e) => {
                                                    if (e.key === 'Enter' && !e.shiftKey) {
                                                        e.preventDefault();
                                                        handleSendMessage();
                                                    }
                                                }}
                                            />

                                            <Button type="button" size="icon" variant="outline"
                                                    onClick={() => messageImageInputRef.current?.click()}>
                                                <Paperclip className="w-4 h-4" />
                                            </Button>
                                            <input
                                                ref={messageImageInputRef}
                                                type="file"
                                                accept="image/*"
                                                multiple
                                                className="hidden"
                                                onChange={onPickMessageFiles}
                                            />

                                            <Button
                                                onClick={handleSendMessage}
                                                disabled={!newMessage.trim() && messageFiles.length === 0}
                                                size="icon"
                                                variant="default"
                                            >
                                                <Send className="w-4 h-4" />
                                            </Button>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>
                        )}
                    </div>

                    {/* Sidebar */}
                    <div className="space-y-6">
                        {/* Status Management */}
                        <Card>
                            <CardHeader>
                                <CardTitle>Status Tiket</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-4">
                                    <div>
                                        <label className="text-sm font-medium text-gray-700 mb-2 block">Status
                                            Semasa</label>
                                        {/* @ts-ignore */}
                                        <Select value={data.status} onValueChange={(value) => setData('status', value)}>
                                            <SelectTrigger>
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="open">Terbuka</SelectItem>
                                                <SelectItem value="in_progress">Dalam Proses</SelectItem>
                                                <SelectItem value="resolved">Selesai</SelectItem>
                                                <SelectItem value="closed">Ditutup</SelectItem>
                                            </SelectContent>
                                        </Select>
                                    </div>

                                    <Button onClick={handleStatusUpdate} disabled={processing} className="w-full">
                                        Kemas Kini Status
                                    </Button>
                                </div>
                            </CardContent>
                        </Card>

                        {/* Ticket Information */}
                        <Card>
                            <CardHeader>
                                <CardTitle>Maklumat Tiket</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-3">
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Keutamaan:</span>
                                        <span
                                            className={cn('text-sm font-medium capitalize', getPriorityColor(ticket.priority))}>
                                            {ticket.priority === 'low' && 'Rendah'}
                                            {ticket.priority === 'medium' && 'Sederhana'}
                                            {ticket.priority === 'high' && 'Tinggi'}
                                        </span>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Status:</span>
                                        <Badge className={getStatusColor(ticket.status)}>
                                            {getStatusText(ticket.status)}
                                        </Badge>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Dicipta:</span>
                                        <span className="text-sm text-gray-900">
                                            {new Date(ticket.created_at).toLocaleString('ms-MY')}
                                        </span>
                                    </div>
                                    {ticket.resolved_at && (
                                        <div className="flex items-center gap-2">
                                            <span className="text-sm font-medium text-gray-600">Diselesaikan:</span>
                                            <span className="text-sm text-gray-900">
                                                {new Date(ticket.resolved_at).toLocaleString('ms-MY')}
                                            </span>
                                        </div>
                                    )}
                                </div>
                            </CardContent>
                        </Card>

                        {/* User Information */}
                        <Card>
                            <CardHeader>
                                <CardTitle>Maklumat Pengguna</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-2">
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Nama:</span>
                                        <span className="text-sm text-gray-900">{ticket.user?.name || 'Unknown'}</span>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Emel:</span>
                                        <span className="text-sm text-gray-900">{ticket.user?.email || 'Unknown'}</span>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Telefon:</span>
                                        <span
                                            className="text-sm text-gray-900">{ticket.user?.phone_number || 'Unknown'}</span>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
