import { Head, Link, router } from '@inertiajs/react';
import { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Input } from '@/components/ui/input';
import {
    Search,
    Filter,
    Eye,
    Trash2,
    MessageCircle,
    Clock,
} from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import Swal from 'sweetalert2';
import { PaginationProps, Ticket } from '@/types';
import { cn } from '@/lib/utils';
import Pagination from '@/components/ui/pagination';
import { toast } from 'react-toastify';
import EmptyData from '@/components/empty-data';

interface TicketIndexProps {
    tickets: PaginationProps<Ticket>;
    filters: {
        search?: string;
        status?: 'all' | 'open' | 'in_progress' | 'resolved' | 'closed';
    };
}

const getStatusColor = (status: string) => {
    switch (status) {
        case 'open':
            return 'text-blue-500 bg-blue-500/20';
        case 'in_progress':
            return 'text-yellow-500 bg-yellow-500/10';
        case 'resolved':
            return 'text-green-500 bg-green-500/10';
        case 'closed':
            return 'text-gray-500 bg-gray-500/10';
        default:
            return 'text-gray-500 bg-gray-500/10';
    }
};

const getStatusText = (status: string) => {
    switch (status) {
        case 'open':
            return 'Terbuka';
        case 'in_progress':
            return 'Dalam Proses';
        case 'resolved':
            return 'Selesai';
        case 'closed':
            return 'Ditutup';
        default:
            return 'Tidak Diketahui';
    }
};

const getPriorityColor = (priority: string) => {
    switch (priority) {
        case 'high':
            return 'text-red-500';
        case 'medium':
            return 'text-yellow-500';
        case 'low':
            return 'text-green-500';
        default:
            return 'text-gray-500';
    }
};

// Helper function to check if ticket has unread messages
const hasUnreadMessages = (ticket: Ticket): boolean => {
    let isUnread = false;

    // Check if ticket has replies
    if (ticket.replies && ticket.replies.length > 0) {
        const lastAdminReply = ticket.replies
            .filter(reply => reply.user?.role === 'admin')
            .sort((a, b) => new Date(b.created_at).getTime() - new Date(a.created_at).getTime())[0];

        const lastUserReply = ticket.replies
            .filter(reply => reply.user?.role !== 'admin')
            .sort((a, b) => new Date(b.created_at).getTime() - new Date(a.created_at).getTime())[0];

        if (lastUserReply) {
            if (lastAdminReply) {
                // If admin has replied, check if user replied after admin's last reply
                if (new Date(lastUserReply.created_at) > new Date(lastAdminReply.created_at)) {
                    isUnread = true;
                }
            } else {
                // If admin hasn't replied yet, check if user replied after ticket was last viewed
                if (ticket.last_viewed_by_admin) {
                    isUnread = new Date(lastUserReply.created_at) > new Date(ticket.last_viewed_by_admin);
                } else {
                    // If never viewed, consider it unread
                    isUnread = true;
                }
            }

            // If ticket is unread, also check if admin has viewed it since the last user reply
            if (isUnread && ticket.last_viewed_by_admin) {
                isUnread = new Date(lastUserReply.created_at) > new Date(ticket.last_viewed_by_admin);
            }
        }
    } else {
        // New ticket with no replies - check if admin has viewed it
        if (!ticket.last_viewed_by_admin) {
            isUnread = true;
        }
    }

    return isUnread;
};

export default function TicketIndex({ tickets, filters }: TicketIndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [status, setStatus] = useState(filters.status || 'all');
    const [previousTicketsCount, setPreviousTicketsCount] = useState(tickets.data.length);

    const handleSearch = () => {
        router.get(route('admin.tickets.index'), {
            search,
            status
        }, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const handleReset = () => {
        setSearch('');
        setStatus('all');
        router.get(route('admin.tickets.index'), {}, {
            preserveState: true,
            preserveScroll: true
        });
    };

    // Check for new tickets and show notifications
    useEffect(() => {
        const currentTicketsCount = tickets.data.length;

        if (currentTicketsCount > previousTicketsCount) {
            // New tickets were added
            const newTickets = tickets.data.slice(previousTicketsCount);

            newTickets.forEach((ticket) => {
                toast.info(`New ticket: "${ticket.title}" by ${ticket.user?.name || 'Unknown User'}`, {
                    position: 'top-right',
                    autoClose: 5000,
                    hideProgressBar: false,
                    closeOnClick: true,
                    pauseOnHover: true,
                    draggable: true
                });
            });

            setPreviousTicketsCount(currentTicketsCount);
        }
    }, [tickets.data, previousTicketsCount]);

    const handleDelete = async (ticket: Ticket) => {
        const result = await Swal.fire({
            title: 'Padam Tiket?',
            text: `Adakah anda pasti mahu memadam tiket "${ticket.title}"?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#dc2626',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'Ya, Padam',
            cancelButtonText: 'Batal'
        });

        if (result.isConfirmed) {
            router.delete(route('admin.tickets.destroy', ticket.id), {
                onSuccess: () => {
                    Swal.fire({
                        title: 'Berjaya!',
                        text: 'Tiket telah dipadam',
                        icon: 'success',
                        timer: 2000
                    });
                    // Also show toast notification
                    toast.success(`Ticket "${ticket.title}" has been deleted successfully`, {
                        position: 'top-right',
                        autoClose: 5000,
                        hideProgressBar: false,
                        closeOnClick: true,
                        pauseOnHover: true,
                        draggable: true
                    });
                },
                onError: () => {
                    Swal.fire({
                        title: 'Ralat!',
                        text: 'Gagal memadam tiket',
                        icon: 'error'
                    });
                    // Also show toast notification
                    toast.error('Failed to delete ticket', {
                        position: 'top-right',
                        autoClose: 5000,
                        hideProgressBar: false,
                        closeOnClick: true,
                        pauseOnHover: true,
                        draggable: true
                    });
                }
            });
        }
    };

    return (
        <AdminLayout>
            <Head title="Pengurusan Tiket" />

            <div className="p-6 space-y-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-3xl font-bold text-gray-900">Pengurusan Tiket</h1>
                        <p className="text-gray-600">Urus semua tiket sokongan pengguna</p>
                        <div className="flex items-center gap-4 mt-2">
                            <div className="flex items-center gap-2">
                                <span className="text-sm text-gray-600">Total Tiket:</span>
                                <Badge variant="outline">{tickets.total}</Badge>
                            </div>
                            <div className="flex items-center gap-2">
                                <span className="text-sm text-gray-600">Unread:</span>
                                <Badge variant="destructive">
                                    {tickets.data.filter(hasUnreadMessages).length}
                                </Badge>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Filters */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                            <Filter className="w-5 h-5" />
                            Penapis
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div className="md:col-span-2">
                                <Input
                                    placeholder="Cari tajuk atau penerangan tiket..."
                                    value={search}
                                    onChange={(e) => setSearch(e.target.value)}
                                    onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
                                />
                            </div>
                            <div>
                                <Select value={status} onValueChange={(value: string) => setStatus(value as any)}>
                                    <SelectTrigger>
                                        <SelectValue placeholder="Status" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">Semua Status</SelectItem>
                                        <SelectItem value="open">Terbuka</SelectItem>
                                        <SelectItem value="in_progress">Dalam Proses</SelectItem>
                                        <SelectItem value="resolved">Selesai</SelectItem>
                                        <SelectItem value="closed">Ditutup</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                        </div>
                        <div className="flex gap-2 mt-4">
                            <Button onClick={handleSearch} className="gap-2">
                                <Search className="w-4 h-4" />
                                Cari
                            </Button>
                            <Button variant="outline" onClick={handleReset}>
                                Reset
                            </Button>
                        </div>
                    </CardContent>
                </Card>

                {/* Tickets Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>Senarai Tiket</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                <tr className="border-b">
                                    <th className="text-left py-3 px-4 font-medium">Tajuk</th>
                                    <th className="text-left py-3 px-4 font-medium">Status</th>
                                    <th className="text-left py-3 px-4 font-medium">Keutamaan</th>
                                    <th className="text-left py-3 px-4 font-medium">Pengguna</th>
                                    <th className="text-left py-3 px-4 font-medium">Balasan</th>
                                    <th className="text-left py-3 px-4 font-medium">Tarikh</th>
                                    <th className="text-left py-3 px-4 font-medium">Unread</th>
                                    <th className="text-left py-3 px-4 font-medium">Tindakan</th>
                                </tr>
                                </thead>
                                <tbody>
                                {tickets.data.map((ticket) => (
                                    <tr key={ticket.id} className="border-b hover:bg-gray-50">
                                        <td className="py-3 px-4">
                                            <div className="max-w-xs">
                                                <div className="flex items-center gap-2">
                                                    <p className="font-medium text-sm">{ticket.title}</p>
                                                    {hasUnreadMessages(ticket) && (
                                                        <div
                                                            className="w-2 h-2 bg-red-500 rounded-full animate-pulse"></div>
                                                    )}
                                                </div>
                                                <p className="text-xs text-gray-500 mt-1 line-clamp-2">
                                                    {ticket.description}
                                                </p>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <Badge
                                                className={cn('text-sm font-medium capitalize', getStatusColor(ticket.status))}>
                                                {getStatusText(ticket.status)}
                                            </Badge>
                                        </td>
                                        <td className="py-3 px-4">
                                                <span
                                                    className={cn('text-sm font-medium capitalize', getPriorityColor(ticket.priority))}>
                                                    {ticket.priority === 'low' && 'Rendah'}
                                                    {ticket.priority === 'medium' && 'Sederhana'}
                                                    {ticket.priority === 'high' && 'Tinggi'}
                                                </span>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="flex items-center">
                                                <span className="text-sm">{ticket.user?.name || 'Unknown'}</span>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="flex items-center gap-1">
                                                <MessageCircle className="w-4 h-4 text-gray-500" />
                                                <span className="text-sm">{ticket.replies?.length || 0}</span>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="flex items-center">
                                                <Clock className="w-4 h-4 mr-2 text-gray-500" />
                                                <span className="text-sm">
                                                        {new Date(ticket.created_at).toLocaleDateString('ms-MY')}
                                                    </span>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="flex items-center justify-center">
                                                {hasUnreadMessages(ticket) ? (
                                                    <div className="flex items-center gap-2">
                                                        <div
                                                            className="w-3 h-3 bg-red-500 rounded-full animate-pulse"></div>
                                                        <span className="text-xs text-red-600 font-medium">Unread</span>
                                                    </div>
                                                ) : (
                                                    <span className="text-xs text-gray-400">-</span>
                                                )}
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="flex flex-row items-center space-x-1">
                                                <Link href={route('admin.tickets.show', ticket.id)}>
                                                    <Button size="icon" variant="outline" title="Lihat tiket">
                                                        <Eye className="w-4 h-4" />
                                                    </Button>
                                                </Link>
                                                <Button
                                                    size="icon"
                                                    variant="outline"
                                                    onClick={() => handleDelete(ticket)}
                                                    className="text-red-600 border-red-600 hover:bg-red-50"
                                                    title="Padam tiket"
                                                >
                                                    <Trash2 className="w-4 h-4" />
                                                </Button>
                                            </div>
                                        </td>
                                    </tr>
                                ))}
                                </tbody>
                            </table>
                            {tickets.data.length === 0 && <EmptyData theme="light" />}
                        </div>

                        {/* Pagination */}
                        <Pagination
                            current_page={tickets.current_page}
                            last_page={tickets.last_page}
                            total={tickets.total}
                            per_page={tickets.per_page}
                            links={tickets.links}
                        />
                    </CardContent>
                </Card>
            </div>
        </AdminLayout>
    );
}
