import { Head, router } from '@inertiajs/react';
import { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Upload, UserPlus, Save } from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import { toast } from 'react-toastify';

interface SystemSettingsProps {
    settings: {
        max_upload_size: number;
        new_user_balance: number;
    };
}

export default function SystemSettings({ settings }: SystemSettingsProps) {
    const [formData, setFormData] = useState({
        max_upload_size: settings.max_upload_size,
        new_user_balance: settings.new_user_balance,
    });
    const [isLoading, setIsLoading] = useState(false);

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setIsLoading(true);

        try {
            router.post(route('admin.system-settings.update'), formData, {
                onSuccess: () => {
                    toast.success('System settings updated successfully!');
                },
                onError: (errors) => {
                    Object.values(errors).forEach(error => {
                        toast.error(error);
                    });
                },
            });
        } catch (error) {
            toast.error('Failed to update system settings');
        } finally {
            setIsLoading(false);
        }
    };

    const handleInputChange = (field: string, value: string | number) => {
        setFormData(prev => ({
            ...prev,
            [field]: value
        }));
    };

    return (
        <AdminLayout>
            <Head title="Tetapan Sistem" />

            <div className="p-6 space-y-6">
                {/* Header */}
                <div className="flex justify-between items-center">
                    <div>
                        <h1 className="text-3xl font-bold text-gray-900">Tetapan Sistem</h1>
                        <p className="text-gray-600">Urus tetapan sistem untuk aplikasi</p>
                    </div>
                </div>

                {/* Settings Form */}
                <form onSubmit={handleSubmit}>
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                        {/* Max Upload Size */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center gap-2">
                                    <Upload className="w-5 h-5 text-blue-600" />
                                    Saiz Muat Naik Maksimum
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div>
                                    <Label htmlFor="max_upload_size">Saiz Fail Maksimum (KB)</Label>
                                    <div className="relative mt-1">
                                        <Input
                                            id="max_upload_size"
                                            type="number"
                                            value={formData.max_upload_size}
                                            onChange={(e) => handleInputChange('max_upload_size', parseInt(e.target.value) || 0)}
                                            min="1"
                                            max="10000"
                                            className="pr-16"
                                        />
                                        <span className="absolute right-3 top-1/2 transform -translate-y-1/2 text-sm text-gray-500">
                                            KB
                                        </span>
                                    </div>
                                    <p className="text-sm text-gray-600 mt-1">
                                        Saiz maksimum fail yang boleh dimuat naik oleh pengguna
                                    </p>
                                    <p className="text-xs text-gray-500 mt-1">
                                        Nilai semasa: {formData.max_upload_size} KB ({(formData.max_upload_size / 1024).toFixed(2)} MB)
                                    </p>
                                </div>
                            </CardContent>
                        </Card>

                        {/* New User Balance */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center gap-2">
                                    <UserPlus className="w-5 h-5 text-green-600" />
                                    Baki Pengguna Baharu
                                </CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div>
                                    <Label htmlFor="new_user_balance">Baki Permulaan (RM)</Label>
                                    <div className="relative mt-1">
                                        <span className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500 font-bold">
                                            RM
                                        </span>
                                        <Input
                                            id="new_user_balance"
                                            type="number"
                                            value={formData.new_user_balance}
                                            onChange={(e) => handleInputChange('new_user_balance', parseFloat(e.target.value) || 0)}
                                            min="0"
                                            max="1000000"
                                            step="0.01"
                                            className="pl-12"
                                        />
                                    </div>
                                    <p className="text-sm text-gray-600 mt-1">
                                        Baki permulaan yang akan diberikan kepada pengguna baharu
                                    </p>
                                    <p className="text-xs text-gray-500 mt-1">
                                        Nilai semasa: RM {formData.new_user_balance.toFixed(2)}
                                    </p>
                                </div>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Submit Button */}
                    <div className="mt-6">
                        <Button
                            type="submit"
                            disabled={isLoading}
                        >
                            <Save className="w-4 h-4 mr-2" />
                            {isLoading ? 'Menyimpan...' : 'Simpan Tetapan'}
                        </Button>
                    </div>
                </form>
            </div>
        </AdminLayout>
    );
}
