import { Head, Link, router } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Plus, Edit, Trash2, Eye, EyeOff } from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import { SystemMessage } from '@/types';
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow
} from '@/components/ui/table';
import Swal from 'sweetalert2';

interface Props {
    messages: SystemMessage[];
}

const Toast = Swal.mixin({
    toast: true,
    position: 'top-end',
    showConfirmButton: false,
    timer: 2500,
    timerProgressBar: true
});

export default function SystemMessagesIndex({ messages }: Props) {
    function handleToggleStatus(message: SystemMessage) {
        router.patch(
            route('admin.system-messages.toggle-status', message.id),
            {}, // <= data kosong (argumen 2 wajib ada kalau mau pakai opsi)
            {
                preserveScroll: true,
                onSuccess: () =>
                    Toast.fire({
                        icon: 'success',
                        title: message.is_active ? 'Mesej dinyahaktifkan.' : 'Mesej diaktifkan.'
                    }),
                onError: () =>
                    Toast.fire({
                        icon: 'error',
                        title: 'Gagal mengubah status.'
                    })
            }
        );
    }

    async function handleDelete(message: SystemMessage) {
        const result = await Swal.fire({
            title: 'Padam mesej ini?',
            text: 'Tindakan ini tidak boleh diundur.',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Ya, padam',
            cancelButtonText: 'Batal',
            reverseButtons: true,
            showLoaderOnConfirm: true,
            allowOutsideClick: () => !Swal.isLoading(),
            preConfirm: () =>
                new Promise<void>((resolve, reject) => {
                    router.delete(
                        route('admin.system-messages.destroy', message.id),
                        {
                            preserveScroll: true,
                            onSuccess: () => resolve(),
                            onError: () => {
                                Swal.showValidationMessage('Gagal memadam mesej.');
                                reject(new Error('delete_failed'));
                            }
                        }
                    );
                })
        });

        if (result.isConfirmed) {
            await Swal.fire({
                icon: 'success',
                title: 'Dipadam!',
                text: 'Mesej telah dipadam.',
                timer: 1800,
                showConfirmButton: false
            });
            // optional: refresh partial props jika pakai server-side props "messages"
            router.reload({ only: ['messages'] });
        }
    }

    return (
        <AdminLayout>
            <Head title="Pengurusan Mesej Sistem" />

            <div className="p-6 space-y-6">
                <div className="flex flex-col md:flex-row items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold text-gray-900">Pengurusan Mesej Sistem</h1>
                        <p className="text-gray-600">Urus mesej sistem yang dipaparkan kepada pengguna</p>
                    </div>
                    <div className='self-end'>
                        <Link href={route('admin.system-messages.create')}>
                            <Button>
                                <Plus className="w-4 h-4 mr-2" />
                                Tambah Mesej
                            </Button>
                        </Link>
                    </div>
                </div>

                {messages.length > 0 ? (
                    <div className="overflow-x-auto rounded-md border">
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead className="w-16">Urut</TableHead>
                                    <TableHead>Tajuk & Mesej</TableHead>
                                    <TableHead className="min-w-[180px]">Pengguna</TableHead>
                                    <TableHead className="w-28">Status</TableHead>
                                    <TableHead className="w-36">Dicipta</TableHead>
                                    <TableHead className="w-56 text-right">Aksi</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {messages.map((message) => (
                                    <TableRow key={message.id}>
                                        <TableCell className="align-top">
                                            <div className="flex items-center gap-2 text-gray-400">
                                                <span className="text-sm font-medium">#{message.display_order}</span>
                                            </div>
                                        </TableCell>

                                        <TableCell className="align-top">
                                            <div className="flex flex-col gap-1 min-w-0">
                                                <div className="flex items-center gap-3">
                                                    <h3 className="font-semibold text-gray-900 truncate">{message.title}</h3>
                                                </div>
                                                <p className="text-sm text-gray-600 line-clamp-2">{message.message}</p>
                                            </div>
                                        </TableCell>

                                        <TableCell className="align-top">
                                            <div className="flex flex-wrap gap-1">
                                                {message.users.length > 0 ? (
                                                    <>
                                                        {message.users.slice(0, 3).map((user: any) => (
                                                            <span
                                                                key={user.id}
                                                                className="text-xs bg-blue-100 text-blue-800 px-2 py-1 rounded"
                                                            >
                                {user.name}
                              </span>
                                                        ))}
                                                        {message.users.length > 3 && (
                                                            <span className="text-xs text-gray-500">
                                +{message.users.length - 3} lagi
                              </span>
                                                        )}
                                                    </>
                                                ) : (
                                                    <span className="text-xs text-gray-400">Tiada pengguna</span>
                                                )}
                                            </div>
                                        </TableCell>

                                        <TableCell className="align-top">
                                            <Badge variant={message.is_active ? 'default' : 'secondary'}>
                                                {message.is_active ? 'Aktif' : 'Tidak Aktif'}
                                            </Badge>
                                        </TableCell>

                                        <TableCell className="align-top text-gray-700">
                                            {new Date(message.created_at).toLocaleDateString('ms-MY')}
                                        </TableCell>

                                        <TableCell className="align-top">
                                            <div className="flex items-center justify-end gap-2 whitespace-nowrap">
                                                <Button
                                                    variant="outline"
                                                    size="sm"
                                                    onClick={() => handleToggleStatus(message)}
                                                >
                                                    {message.is_active ? (
                                                        <>
                                                            <EyeOff className="w-4 h-4 mr-2" />
                                                            Nyahaktifkan
                                                        </>
                                                    ) : (
                                                        <>
                                                            <Eye className="w-4 h-4 mr-2" />
                                                            Aktifkan
                                                        </>
                                                    )}
                                                </Button>

                                                <Link href={route('admin.system-messages.edit', message.id)}>
                                                    <Button variant="outline" size="sm">
                                                        <Edit className="w-4 h-4" />
                                                    </Button>
                                                </Link>

                                                <Button
                                                    variant="outline"
                                                    size="sm"
                                                    onClick={() => handleDelete(message)}
                                                    className="text-red-600 hover:text-red-700"
                                                >
                                                    <Trash2 className="w-4 h-4" />
                                                </Button>
                                            </div>
                                        </TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                        </Table>
                    </div>
                ) : (
                    <div className="text-center py-12">
                        <div className="text-gray-400 mb-4">
                            <svg
                                className="mx-auto h-12 w-12"
                                fill="none"
                                viewBox="0 0 24 24"
                                stroke="currentColor"
                            >
                                <path
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                    strokeWidth={2}
                                    d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"
                                />
                            </svg>
                        </div>
                        <h3 className="text-lg font-medium text-gray-900 mb-2">Tiada Mesej Sistem</h3>
                        <p className="text-gray-600 mb-4">Mulakan dengan menambah mesej sistem untuk pengguna.</p>
                        <Link href={route('admin.system-messages.create')}>
                            <Button>
                                <Plus className="w-4 h-4 mr-2" />
                                Tambah Mesej Pertama
                            </Button>
                        </Link>
                    </div>
                )}
            </div>
        </AdminLayout>
    );
}
