import { Head, router } from '@inertiajs/react';
import { ArrowLeft, Search } from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Checkbox } from '@/components/ui/checkbox';
import { useState } from 'react';

interface SystemMessage {
    id: number;
    title: string;
    message: string;
    is_active: boolean;
    display_order: number;
    users: User[];
}

interface User {
    id: number;
    name: string;
    email: string;
}

interface Props {
    message: SystemMessage;
    users: User[];
}

export default function EditSystemMessage({ message, users }: Props) {
    const [formData, setFormData] = useState({
        title: message.title,
        message: message.message,
        display_order: message.display_order.toString(),
        user_ids: message.users.map(user => user.id)
    });

    const [searchTerm, setSearchTerm] = useState('');

    // Filter users based on search term
    const filteredUsers = users.filter(user =>
        user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        user.email.toLowerCase().includes(searchTerm.toLowerCase())
    );

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();

        const data = {
            title: formData.title,
            message: formData.message,
            display_order: formData.display_order ? parseInt(formData.display_order) : message.display_order,
            user_ids: formData.user_ids
        };

        router.put(route('admin.system-messages.update', message.id), data);
    };

    const handleUserToggle = (userId: number) => {
        setFormData(prev => ({
            ...prev,
            user_ids: prev.user_ids.includes(userId)
                ? prev.user_ids.filter(id => id !== userId)
                : [...prev.user_ids, userId]
        }));
    };

    return (
        <AdminLayout>
            <Head title="Edit Mesej Sistem" />

            <div className="space-y-6 p-6">
                <div className="flex items-center gap-4">
                    <Button
                        variant="outline"
                        onClick={() => router.get(route('admin.system-messages.index'))}
                    >
                        <ArrowLeft className="w-4 h-4 mr-2" />
                        Kembali
                    </Button>
                    <div>
                        <h1 className="text-2xl font-bold text-gray-900">Edit Mesej Sistem</h1>
                        <p className="text-gray-600">Kemas kini maklumat mesej sistem</p>
                    </div>
                </div>

                <Card>
                    <CardHeader>
                        <CardTitle>Maklumat Mesej</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <form onSubmit={handleSubmit} className="space-y-6">
                            <div className="space-y-2">
                                <Label htmlFor="title">Tajuk Mesej</Label>
                                <Input
                                    id="title"
                                    value={formData.title}
                                    onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                                    placeholder="Masukkan tajuk mesej"
                                    required
                                />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="message">Kandungan Mesej</Label>
                                <Textarea
                                    id="message"
                                    value={formData.message}
                                    onChange={(e) => setFormData({ ...formData, message: e.target.value })}
                                    placeholder="Masukkan kandungan mesej"
                                    rows={4}
                                    required
                                />
                                <p className="text-sm text-gray-500">
                                    Maksimum 1000 aksara
                                </p>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="display_order">Susunan Paparan</Label>
                                <Input
                                    id="display_order"
                                    type="number"
                                    value={formData.display_order}
                                    onChange={(e) => setFormData({ ...formData, display_order: e.target.value })}
                                    placeholder="Susunan paparan"
                                    min="1"
                                    required
                                />
                                <p className="text-sm text-gray-500">
                                    Nombor yang lebih kecil akan dipaparkan terlebih dahulu
                                </p>
                            </div>

                            <div className="space-y-2">
                                <Label>Pilih Pengguna</Label>
                                <div className="space-y-3">
                                    {/* Search Input */}
                                    <div className="relative">
                                        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                                        <Input
                                            type="text"
                                            placeholder="Cari pengguna mengikut nama atau e-mel..."
                                            value={searchTerm}
                                            onChange={(e) => setSearchTerm(e.target.value)}
                                            className="pl-10"
                                        />
                                    </div>

                                    {/* User List */}
                                    <div className="max-h-60 overflow-y-auto border rounded-lg p-4 space-y-2">
                                        {filteredUsers.length > 0 ? (
                                            filteredUsers.map((user) => (
                                                <div key={user.id} className="flex items-center space-x-3">
                                                    <Checkbox
                                                        id={`user-${user.id}`}
                                                        checked={formData.user_ids.includes(user.id)}
                                                        onCheckedChange={() => handleUserToggle(user.id)}
                                                    />
                                                    <Label htmlFor={`user-${user.id}`} className="text-sm">
                                                        {user.name} ({user.email})
                                                    </Label>
                                                </div>
                                            ))
                                        ) : (
                                            <div className="text-center py-4 text-gray-500">
                                                {searchTerm ? 'Tiada pengguna dijumpai' : 'Tiada pengguna tersedia'}
                                            </div>
                                        )}
                                    </div>

                                    {/* Selected Users Count */}
                                    {formData.user_ids.length > 0 && (
                                        <div className="text-sm text-blue-600 font-medium">
                                            {formData.user_ids.length} pengguna dipilih
                                        </div>
                                    )}
                                </div>
                                <p className="text-sm text-gray-500">
                                    Pilih pengguna yang akan melihat mesej ini
                                </p>
                            </div>

                            <div className="flex justify-end gap-4">
                                <Button
                                    type="button"
                                    variant="outline"
                                    onClick={() => router.get(route('admin.system-messages.index'))}
                                >
                                    Batal
                                </Button>
                                <Button type="submit">
                                    Kemas Kini Mesej
                                </Button>
                            </div>
                        </form>
                    </CardContent>
                </Card>
            </div>
        </AdminLayout>
    );
}
