import { Head, Link } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { ArrowLeft, Edit, Calendar, User, FileText } from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import { NewsProps } from '@/types';
import { formatDate } from '@/lib/utils';

interface NewsShowProps {
    news: NewsProps;
}

export default function NewsShow({ news }: NewsShowProps) {
    return (
        <AdminLayout>
            <Head title={news.title} />

            <div className="p-6 space-y-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div className="flex items-center gap-4">
                        <Link href={route('admin.news.index')}>
                            <Button variant="outline" size="icon">
                                <ArrowLeft className="w-4 h-4" />
                            </Button>
                        </Link>
                        <div>
                            <h1 className="text-3xl font-bold text-gray-900">{news.title}</h1>
                            <p className="text-gray-600">Butiran berita</p>
                        </div>
                    </div>
                    <Link href={route('admin.news.edit', news.id)}>
                        <Button>
                            <Edit className="w-4 h-4 mr-2" />
                            Edit Berita
                        </Button>
                    </Link>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    {/* Main Content */}
                    <div className="lg:col-span-2 space-y-6">
                        {/* Image */}
                        {news.image && (
                            <Card>
                                <CardContent className="p-0">
                                    <img
                                        src={`/storage/${news.image}`}
                                        alt={news.title}
                                        className="w-full h-64 object-cover rounded-t-lg"
                                    />
                                </CardContent>
                            </Card>
                        )}

                        {/* Description */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center gap-2">
                                    <FileText className="w-5 h-5" />
                                    Keterangan Berita
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="prose max-w-none">
                                    <div className="whitespace-pre-wrap text-gray-700 leading-relaxed">
                                        {news.description}
                                    </div>
                                </div>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Sidebar */}
                    <div className="space-y-6">
                        {/* Author Information */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center gap-2">
                                    <User className="w-5 h-5" />
                                    Maklumat Pengarang
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-2">
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Nama:</span>
                                        <span className="text-sm text-gray-900">{news.user?.name || 'Unknown'}</span>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Emel:</span>
                                        <span className="text-sm text-gray-900">{news.user?.email || 'Unknown'}</span>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        {/* Creation Information */}
                        <Card>
                            <CardHeader>
                                <CardTitle className="flex items-center gap-2">
                                    <Calendar className="w-5 h-5" />
                                    Maklumat Penciptaan
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-3">
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Dicipta pada:</span>
                                        <span className="text-sm text-gray-900">{formatDate(news.created_at)}</span>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <span className="text-sm font-medium text-gray-600">Dikemas kini pada:</span>
                                        <span className="text-sm text-gray-900">{formatDate(news.updated_at)}</span>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        {/* Actions */}
                        <Card>
                            <CardContent className="pt-6">
                                <div className="space-y-3">
                                    <Link href={route('admin.news.edit', news.id)}>
                                        <Button className="w-full">
                                            <Edit className="w-4 h-4 mr-2" />
                                            Edit Berita
                                        </Button>
                                    </Link>

                                    <Link href={route('admin.news.index')}>
                                        <Button variant="outline" className="w-full">
                                            Kembali ke Senarai
                                        </Button>
                                    </Link>
                                </div>
                            </CardContent>
                        </Card>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
