import { Head, Link, router } from '@inertiajs/react';
import { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import {
    Plus,
    Search,
    Filter,
    Edit,
    Eye,
    Trash2,
    User
} from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import Swal from 'sweetalert2';
import { PaginationProps, NewsProps } from '@/types';
import Pagination from '@/components/ui/pagination';
import EmptyData from '@/components/empty-data';

interface NewsIndexProps {
    news: PaginationProps<NewsProps>;
    filters: {
        search?: string;
    };
}

export default function NewsIndex({ news, filters }: NewsIndexProps) {
    const [search, setSearch] = useState(filters.search || '');

    const handleSearch = () => {
        router.get(route('admin.news.index'), {
            search
        }, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const handleReset = () => {
        setSearch('');
        router.get(route('admin.news.index'), {}, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const handleDelete = async (newsItem: NewsProps) => {
        const result = await Swal.fire({
            title: 'Padam Berita?',
            text: `Adakah anda pasti mahu memadam berita "${newsItem.title}"?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#ef4444',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'Ya, Padam',
            cancelButtonText: 'Batal'
        });

        if (result.isConfirmed) {
            router.delete(route('admin.news.destroy', newsItem.id), {
                onSuccess: () => {
                    Swal.fire({
                        title: 'Berjaya!',
                        text: 'Berita telah dipadam',
                        icon: 'success',
                        timer: 2000
                    });
                },
                onError: () => {
                    Swal.fire({
                        title: 'Ralat!',
                        text: 'Gagal memadam berita',
                        icon: 'error'
                    });
                }
            });
        }
    };


    return (
        <AdminLayout>
            <Head title="Pengurusan Berita" />

            <div className="p-6 space-y-6">
                {/* Header */}
                <div className="flex justify-between items-center">
                    <div>
                        <h1 className="text-3xl font-bold text-gray-900">Pengurusan Berita</h1>
                        <p className="text-gray-600">Urus semua berita dalam sistem</p>
                    </div>
                    <Link href={route('admin.news.create')}>
                        <Button>
                            <Plus className="w-4 h-4 mr-2" />
                            Tambah Berita
                        </Button>
                    </Link>
                </div>

                {/* Filters */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center">
                            <Filter className="w-4 h-4 mr-2" />
                            Penapis
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="grid grid-cols-1 md:grid-cols-1 gap-4">
                            <div>
                                <Input
                                    placeholder="Cari tajuk atau keterangan berita..."
                                    value={search}
                                    onChange={(e) => setSearch(e.target.value)}
                                    onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
                                />
                            </div>
                        </div>
                        <div className="flex gap-2 mt-4">
                            <Button onClick={handleSearch} variant="outline">
                                <Search className="w-4 h-4 mr-2" />
                                Cari
                            </Button>
                            <Button onClick={handleReset} variant="outline">
                                Reset
                            </Button>
                        </div>
                    </CardContent>
                </Card>

                {/* News Table */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center justify-between">
                            <span>Senarai Berita ({news.total})</span>
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                <tr className="border-b">
                                    <th className="text-left py-3 px-4 font-medium">Gambar</th>
                                    <th className="text-left py-3 px-4 font-medium">Tajuk</th>
                                    <th className="text-left py-3 px-4 font-medium">Keterangan</th>
                                    <th className="text-left py-3 px-4 font-medium">Pengarang</th>
                                    <th className="text-left py-3 px-4 font-medium">Tindakan</th>
                                </tr>
                                </thead>
                                <tbody>
                                {news.data.map((newsItem) => (
                                    <tr key={newsItem.id} className="border-b hover:bg-gray-50">
                                        <td className="py-3 px-4">
                                            {newsItem.image ? (
                                                <img
                                                    src={`/storage/${newsItem.image}`}
                                                    alt={newsItem.title}
                                                    className="w-16 h-16 object-cover rounded-lg"
                                                />
                                            ) : (
                                                <div
                                                    className="w-16 h-16 bg-gray-200 rounded-lg flex items-center text-center justify-center">
                                                    <span className="text-xs text-gray-500">Tiada Gambar</span>
                                                </div>
                                            )}
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="max-w-xs">
                                                <p className="font-medium text-sm">{newsItem.title}</p>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="max-w-xs">
                                                <p className="text-xs text-gray-600">
                                                    {newsItem.description.substring(0, 100)}...
                                                </p>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="flex items-center">
                                                <User className="w-4 h-4 mr-2 text-gray-500" />
                                                <span className="text-sm">{newsItem.user?.name || 'Unknown'}</span>
                                            </div>
                                        </td>
                                        <td className="py-3 px-4">
                                            <div className="flex flex-row items-center space-x-1">
                                                <Link href={route('admin.news.show', newsItem.id)}>
                                                    <Button size="icon" variant="outline" title="Lihat berita">
                                                        <Eye className="w-4 h-4" />
                                                    </Button>
                                                </Link>
                                                <Link href={route('admin.news.edit', newsItem.id)}>
                                                    <Button size="icon" variant="outline" title="Edit berita">
                                                        <Edit className="w-4 h-4" />
                                                    </Button>
                                                </Link>
                                                <Button
                                                    size="icon"
                                                    variant="outline"
                                                    onClick={() => handleDelete(newsItem)}
                                                    className="text-red-600 border-red-600 hover:bg-red-50"
                                                    title="Padam berita"
                                                >
                                                    <Trash2 className="w-4 h-4" />
                                                </Button>
                                            </div>
                                        </td>
                                    </tr>
                                ))}
                                </tbody>
                            </table>
                            {news.data.length === 0 && <EmptyData theme="light" />}
                        </div>

                        {/* Pagination */}
                        <Pagination
                            current_page={news.current_page}
                            last_page={news.last_page}
                            total={news.total}
                            per_page={news.per_page}
                            links={news.links}
                        />
                    </CardContent>
                </Card>
            </div>
        </AdminLayout>
    );
}
