import { Head, Link, useForm } from '@inertiajs/react';
import { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { ArrowLeft, Save, Upload, X } from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import Swal from 'sweetalert2';
import { NewsProps } from '@/types';

interface NewsEditProps {
    news: NewsProps;
}

export default function NewsEdit({ news }: NewsEditProps) {
    const [imagePreview, setImagePreview] = useState<string | null>(null);
    const [isSubmitting, setIsSubmitting] = useState(false);

    // ✔ useForm state
    const { data, setData, post, processing, errors, transform } = useForm({
        title: news.title ?? '',
        description: news.description ?? '',
        image: null as File | null,
    });

    useEffect(() => {
        setImagePreview(news.image ? `/storage/${news.image}` : null);
    }, [news.image]);

    const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0] || null;
        setData('image', file);
        if (file) {
            const reader = new FileReader();
            reader.onload = ev => setImagePreview(ev.target?.result as string);
            reader.readAsDataURL(file);
        } else {
            setImagePreview(null);
        }
    };

    const removeImage = () => {
        setData('image', null);
        setImagePreview(null);
    };

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        setIsSubmitting(true);

        // ✔ Method override + force multipart
        transform(form => ({
            ...form,
            _method: 'put',
        }));

        post(route('admin.news.update', news.id), {
            forceFormData: true,          // <-- important
            preserveScroll: true,
            onSuccess: () => {
                Swal.fire({
                    title: 'Berjaya!',
                    text: 'Berita telah dikemas kini',
                    icon: 'success',
                    timer: 2000,
                    showConfirmButton: false,
                });
            },
            onError: () => {
                Swal.fire({
                    title: 'Ralat!',
                    text: 'Gagal mengemas kini berita',
                    icon: 'error',
                });
            },
            onFinish: () => setIsSubmitting(false),
        });
    };

    return (
        <AdminLayout>
            <Head title="Edit Berita" />

            <div className="p-6 space-y-6">
                <div className="flex items-center gap-4">
                    <Link href={route('admin.news.index')}>
                        <Button variant="outline" size="icon">
                            <ArrowLeft className="w-4 h-4" />
                        </Button>
                    </Link>
                    <div>
                        <h1 className="text-3xl font-bold text-gray-900">Edit Berita</h1>
                        <p className="text-gray-600">Kemas kini maklumat berita</p>
                    </div>
                </div>

                {/* encType won’t hurt, but Inertia ignores it — the important part is forceFormData above */}
                <form onSubmit={handleSubmit} className="space-y-6" encType="multipart/form-data">
                    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                        <div className="lg:col-span-2 space-y-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Tajuk Berita</CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-2">
                                        <Label htmlFor="title">Tajuk *</Label>
                                        <Input
                                            id="title"
                                            value={data.title}
                                            onChange={(e) => setData('title', e.target.value)}
                                            placeholder="Masukkan tajuk berita..."
                                            className={errors.title ? 'border-red-500' : ''}
                                        />
                                        {errors.title && <p className="text-sm text-red-500">{errors.title}</p>}
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Keterangan Berita</CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-2">
                                        <Label htmlFor="description">Keterangan *</Label>
                                        <Textarea
                                            id="description"
                                            value={data.description}
                                            onChange={(e) => setData('description', e.target.value)}
                                            placeholder="Masukkan keterangan berita..."
                                            rows={6}
                                            maxLength={1000}
                                            className={errors.description ? 'border-red-500' : ''}
                                        />
                                        <div className="text-xs text-gray-500 text-right">
                                            {data.description.length}/1000
                                        </div>
                                        {errors.description && <p className="text-sm text-red-500">{errors.description}</p>}
                                    </div>
                                </CardContent>
                            </Card>
                        </div>

                        <div className="space-y-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Gambar Berita</CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-4">
                                        {imagePreview ? (
                                            <div className="relative">
                                                <img src={imagePreview} alt="Preview" className="w-full h-48 object-cover rounded-lg" />
                                                <Button
                                                    type="button"
                                                    variant="outline"
                                                    size="icon"
                                                    onClick={removeImage}
                                                    className="absolute top-2 right-2 bg-white/80 hover:bg-white"
                                                >
                                                    <X className="w-4 h-4" />
                                                </Button>
                                            </div>
                                        ) : (
                                            <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                                                <Upload className="w-8 h-8 text-gray-400 mx-auto mb-2" />
                                                <p className="text-sm text-gray-600">Tiada gambar dipilih</p>
                                            </div>
                                        )}

                                        <div className="space-y-2">
                                            <Label htmlFor="image">Pilih Gambar</Label>
                                            <Input
                                                id="image"
                                                name="image"
                                                type="file"
                                                accept="image/*"
                                                onChange={handleImageChange}
                                                className="cursor-pointer"
                                            />
                                            <p className="text-xs text-gray-500">
                                                Format: JPEG, PNG, JPG, GIF, SVG. Saiz maksimum: 2MB
                                            </p>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardContent className="pt-6">
                                    <div className="space-y-3">
                                        <Button type="submit" className="w-full" disabled={processing || isSubmitting}>
                                            {processing || isSubmitting ? (
                                                <div className="flex items-center">
                                                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                                    Menyimpan...
                                                </div>
                                            ) : (
                                                <div className="flex items-center">
                                                    <Save className="w-4 h-4 mr-2" />
                                                    Simpan Perubahan
                                                </div>
                                            )}
                                        </Button>

                                        <Link href={route('admin.news.index')}>
                                            <Button variant="outline" className="w-full">Batal</Button>
                                        </Link>
                                    </div>
                                </CardContent>
                            </Card>
                        </div>
                    </div>
                </form>
            </div>
        </AdminLayout>
    );
}
