import { Head } from '@inertiajs/react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import {
    Users,
    CreditCard,
    AlertCircle,
    CheckCircle,
    Clock,
    XCircle,
    Calendar
} from 'lucide-react';
import { Link } from '@inertiajs/react';
import AdminLayout from '@/layouts/admin-layout';
import { formatAmount } from '@/lib/utils';

interface DashboardProps {
    stats: {
        total_users: number;
        total_transactions: number;
        pending_transactions: number;
        processing_transactions: number;
        completed_transactions: number;
        failed_transactions: number;
        total_amount: number;
        today_transactions: number;
    };
    recent_transactions: Array<{
        id: number;
        type: string;
        amount: number;
        status: string;
        reference: string;
        recipient: string;
        created_at: string;
        user: {
            name: string;
            email: string;
        };
    }>;
    recent_users: Array<{
        id: number;
        name: string;
        email: string;
        role: string;
        created_at: string;
    }>;
    transactions_by_type: Array<{
        type: string;
        count: number;
    }>;
    transactions_by_status: Array<{
        status: string;
        count: number;
    }>;
}

export default function AdminDashboard({
                                           stats,
                                           recent_transactions,
                                           recent_users,
                                           transactions_by_type,
                                           transactions_by_status
                                       }: DashboardProps) {
    const getStatusColor = (status: string) => {
        switch (status) {
            case 'completed':
                return 'bg-green-100 text-green-800';
            case 'processing':
                return 'bg-yellow-100 text-yellow-800';
            case 'failed':
                return 'bg-red-100 text-red-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const getStatusLabel = (status: string) => {
        switch (status) {
            case 'completed':
                return 'Selesai';
            case 'processing':
                return 'Dalam Proses';
            case 'failed':
                return 'Gagal';
            default:
                return 'Menunggu';
        }
    };

    const formatDate = (dateString: string) => {
        return new Date(dateString).toLocaleDateString('ms-MY', {
            day: '2-digit',
            month: 'short',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    return (
        <AdminLayout>
            <Head title="Admin Dashboard" />

            <div className="p-6 space-y-6">
                {/* Header */}
                <div className="flex justify-between items-center">
                    <h1 className="text-3xl font-bold text-gray-900">Dashboard</h1>
                </div>

                {/* Stats Grid */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Jumlah Pengguna</CardTitle>
                            <Users className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total_users}</div>
                            <p className="text-xs text-muted-foreground">Total pengguna berdaftar</p>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Jumlah Transaksi</CardTitle>
                            <CreditCard className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total_transactions}</div>
                            <p className="text-xs text-muted-foreground">Total transaksi keseluruhan</p>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Transaksi Hari Ini</CardTitle>
                            <Calendar className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.today_transactions}</div>
                            <p className="text-xs text-muted-foreground">Transaksi hari ini</p>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Jumlah Keseluruhan</CardTitle>
                            <span className="text-muted-foreground">RM</span>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{formatAmount(stats.total_amount)}</div>
                            <p className="text-xs text-muted-foreground">Jumlah transaksi berjaya</p>
                        </CardContent>
                    </Card>
                </div>

                {/* Transaction Status Overview */}
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <Card>
                        <CardContent className="p-4">
                            <div className="flex items-center space-x-2">
                                <Clock className="h-5 w-5 text-yellow-600" />
                                <div>
                                    <p className="text-sm font-medium text-gray-600">Menunggu</p>
                                    <p className="text-2xl font-bold text-yellow-600">{stats.pending_transactions}</p>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardContent className="p-4">
                            <div className="flex items-center space-x-2">
                                <AlertCircle className="h-5 w-5 text-blue-600" />
                                <div>
                                    <p className="text-sm font-medium text-gray-600">Dalam Proses</p>
                                    <p className="text-2xl font-bold text-blue-600">{stats.processing_transactions}</p>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardContent className="p-4">
                            <div className="flex items-center space-x-2">
                                <CheckCircle className="h-5 w-5 text-green-600" />
                                <div>
                                    <p className="text-sm font-medium text-gray-600">Selesai</p>
                                    <p className="text-2xl font-bold text-green-600">{stats.completed_transactions}</p>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardContent className="p-4">
                            <div className="flex items-center space-x-2">
                                <XCircle className="h-5 w-5 text-red-600" />
                                <div>
                                    <p className="text-sm font-medium text-gray-600">Gagal</p>
                                    <p className="text-2xl font-bold text-red-600">{stats.failed_transactions}</p>
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                </div>

                {/* Recent Activity */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 md:gap-6">
                    {/* Recent Transactions */}
                    <Card>
                        <CardHeader>
                            <CardTitle className="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
                                <span className="text-base sm:text-lg">Transaksi Terkini</span>
                                <Link href={route('admin.transactions.index')}>
                                    <Button variant="outline" size="sm" className="w-full sm:w-auto">
                                        Lihat Semua
                                    </Button>
                                </Link>
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="space-y-3 sm:space-y-4">
                                {recent_transactions.map((transaction) => (
                                    <div
                                        key={transaction.id}
                                        className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-2 p-3 sm:p-4 bg-primary/10 rounded-lg overflow-hidden"
                                    >
                                        <div className="flex-1 min-w-0">
                                            <div className="flex flex-wrap items-center gap-2">
                                                <span className="font-medium truncate">{transaction.reference}</span>
                                                <Badge className={`shrink-0 ${getStatusColor(transaction.status)}`}>
                                                    {getStatusLabel(transaction.status)}
                                                </Badge>
                                            </div>
                                            <p className="text-sm text-gray-600 truncate">
                                                {transaction.type} - {transaction.recipient}
                                            </p>
                                            <p className="text-xs text-gray-500 truncate">
                                                {transaction.user.name} • {formatDate(transaction.created_at)}
                                            </p>
                                        </div>

                                        <div className="sm:text-right text-left">
                                            <p
                                                className={`font-bold whitespace-nowrap ${
                                                    transaction.amount > 0 ? 'text-green-600' : 'text-red-600'
                                                }`}
                                            >
                                                {formatAmount(transaction.amount)}
                                            </p>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </CardContent>
                    </Card>

                    {/* Recent Users */}
                    <Card>
                        <CardHeader>
                            <CardTitle className="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
                                <span className="text-base sm:text-lg">Pengguna Terkini</span>
                                <Link href={route('admin.users.index')}>
                                    <Button variant="outline" size="sm" className="w-full sm:w-auto">
                                        Lihat Semua
                                    </Button>
                                </Link>
                            </CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="space-y-3 sm:space-y-4">
                                {recent_users.map((user) => (
                                    <div
                                        key={user.id}
                                        className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-2 p-3 sm:p-4 bg-primary/10 rounded-lg overflow-hidden"
                                    >
                                        <div className="flex-1 min-w-0">
                                            <p className="font-medium truncate">{user.name}</p>
                                            <p className="text-sm text-gray-600 truncate">{user.email}</p>
                                            <p className="text-xs text-gray-500 truncate">Dicipta {formatDate(user.created_at)}</p>
                                        </div>

                                        <div className="sm:text-right text-left">
                                            <Badge
                                                className="shrink-0"
                                                variant={user.role === 'admin' ? 'destructive' : 'secondary'}
                                            >
                                                {user.role === 'admin' ? 'Admin' : 'Pengguna'}
                                            </Badge>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </CardContent>
                    </Card>
                </div>


                {/* Charts Section */}
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {/* Transactions by Type */}
                    <Card>
                        <CardHeader>
                            <CardTitle>Transaksi Mengikut Jenis</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="space-y-3">
                                {transactions_by_type.map((item) => (
                                    <div key={item.type} className="flex items-center justify-between">
                                        <span className="text-sm font-medium">{item.type}</span>
                                        <div className="flex items-center space-x-2">
                                            <div className="w-32 bg-gray-200 rounded-full h-2">
                                                <div
                                                    className="bg-blue-600 h-2 rounded-full"
                                                    style={{ width: `${(item.count / stats.total_transactions) * 100}%` }}
                                                ></div>
                                            </div>
                                            <span className="text-sm font-medium w-8">{item.count}</span>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </CardContent>
                    </Card>

                    {/* Transactions by Status */}
                    <Card>
                        <CardHeader>
                            <CardTitle>Transaksi Mengikut Status</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="space-y-3">
                                {transactions_by_status.map((item) => (
                                    <div key={item.status} className="flex items-center justify-between">
                                        <span className="text-sm font-medium">{getStatusLabel(item.status)}</span>
                                        <div className="flex items-center space-x-2">
                                            <div className="w-32 bg-gray-200 rounded-full h-2">
                                                <div
                                                    className={`h-2 rounded-full ${
                                                        item.status === 'completed' ? 'bg-green-600' :
                                                            item.status === 'processing' ? 'bg-yellow-600' :
                                                                item.status === 'failed' ? 'bg-red-600' : 'bg-gray-600'
                                                    }`}
                                                    style={{ width: `${(item.count / stats.total_transactions) * 100}%` }}
                                                ></div>
                                            </div>
                                            <span className="text-sm font-medium w-8">{item.count}</span>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </AdminLayout>
    );
}
