import { Head, router } from '@inertiajs/react';
import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Pencil, Plus, Trash } from 'lucide-react';
import AdminLayout from '@/layouts/admin-layout';
import BankForm from './form';
import { Bank, PaginationProps } from '@/types';
import Pagination from '@/components/ui/pagination';
import Swal from 'sweetalert2';
import EmptyData from '@/components/empty-data';

type Props = {
    banks: PaginationProps<Bank>
    flash: {
        success: string;
    }
};


export default function MasterBank({ banks, ...pageProps }: Props) {
    const [isFormOpen, setIsFormOpen] = useState(false);
    const [selectedBank, setSelectedBank] = useState<Bank | null>(null);
    const { flash } = pageProps;

    const handleEdit = (bank: Bank) => {
        setSelectedBank(bank);
        setIsFormOpen(true);
    };

    const handleAddNew = () => {
        setSelectedBank(null);
        setIsFormOpen(true);
    };

    const handleCloseForm = () => {
        setIsFormOpen(false);
        setSelectedBank(null);
    };

    const confirmDelete = (bank: Bank) => {
        Swal.fire({
            title: 'Delete bank?',
            html: `This will permanently remove <b>${bank.name}</b>.`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes, delete',
            cancelButtonText: 'Cancel',
            reverseButtons: true
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('admin.banks.destroy', bank.id), {
                    preserveScroll: true,
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Deleted!',
                            text: 'Bank has been deleted.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false
                        });
                    },
                    onError: () => {
                        Swal.fire('Error', 'Failed to delete bank.', 'error');
                    }
                });
            }
        });
    };

    return (
        <AdminLayout>
            <Head title="Manage Banks" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {flash?.success && (
                        <div className="mb-4 p-4 bg-green-100 text-green-700 rounded">
                            {flash.success}
                        </div>
                    )}

                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6 bg-white border-b border-gray-200">
                            <div className="flex justify-between items-center mb-6">
                                <h2 className="text-2xl font-semibold">Manage Banks</h2>
                                <Button variant="primary" onClick={handleAddNew}>
                                    <Plus className="h-4 w-4 mr-2" />
                                    Add New Bank
                                </Button>
                            </div>

                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                    <tr>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            Logo
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            Name
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            Status
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            Created
                                        </th>
                                        <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            Actions
                                        </th>
                                    </tr>
                                    </thead>
                                    <tbody className="bg-white divide-y divide-gray-200">
                                    {banks.data.map((bank) => (
                                        <tr key={bank.id} className="hover:bg-gray-50">
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                {bank.logo ? (
                                                    <img
                                                        src={bank.logo}
                                                        alt={bank.name}
                                                        className="h-10 w-10 rounded-full object-cover"
                                                    />
                                                ) : (
                                                    <div
                                                        className="h-10 w-10 rounded-full bg-gray-200 flex items-center justify-center">
                                                        <span className="text-gray-500 text-xs">No Logo</span>
                                                    </div>
                                                )}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="font-medium text-gray-900">{bank.name}</div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                    <span
                                                        className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${bank.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                                                        {bank.is_active ? 'Active' : 'Inactive'}
                                                    </span>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                                {new Date(bank.created_at).toLocaleDateString()}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium space-x-2">
                                                <button
                                                    onClick={() => handleEdit(bank)}
                                                    className="text-indigo-600 hover:text-indigo-900"
                                                    title="Edit"
                                                >
                                                    <Pencil className="h-5 w-5" />
                                                </button>
                                                <button
                                                    onClick={() => confirmDelete(bank)}
                                                    className="text-red-600 hover:text-red-900"
                                                    title="Delete"
                                                >
                                                    <Trash className="h-5 w-5" />
                                                </button>
                                            </td>
                                        </tr>
                                    ))}
                                    </tbody>
                                </table>
                                {banks.data.length === 0 && <EmptyData theme="light" />}
                            </div>

                            {/* Pagination */}
                            <Pagination
                                current_page={banks.current_page}
                                last_page={banks.last_page}
                                total={banks.total}
                                per_page={banks.per_page}
                                links={banks.links}
                            />
                        </div>
                    </div>
                </div>
            </div>

            <BankForm
                key={selectedBank?.id ?? 'create'}
                bank={selectedBank || undefined}
                isOpen={isFormOpen}
                onClose={handleCloseForm}
            />

        </AdminLayout>
    );
}
