import { router, useForm } from '@inertiajs/react';
import { useEffect, useMemo, useRef } from 'react';
import { Bank } from '@/types';
import { Input } from '@/components/ui/input';
import FileInput from '@/components/ui/file-input';
import { Checkbox } from '@/components/ui/checkbox';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import {
    Sheet, SheetContent, SheetHeader, SheetTitle, SheetDescription, SheetFooter, SheetClose
} from '@/components/ui/sheet';

interface BankFormProps {
    bank?: Bank;
    onClose: () => void;
    isOpen: boolean;
}

type BankFormState = {
    name: string;
    is_active: boolean;
    logo: File | string | null; // allow DB string
};


export default function BankForm({ bank, onClose, isOpen }: BankFormProps) {
    const { data, setData, processing, errors, reset } = useForm({
        name: bank?.name ?? '',
        is_active: bank?.is_active ?? true,
        logo: bank?.logo ?? null as File | string | null,
    });

    // Build preview URL from either File or string
    const objectUrlRef = useRef<string | null>(null);
    const previewUrl = useMemo(() => {
        // Clean up previous File object URL
        if (objectUrlRef.current) {
            URL.revokeObjectURL(objectUrlRef.current);
            objectUrlRef.current = null;
        }

        if (data.logo instanceof File) {
            const url = URL.createObjectURL(data.logo);
            objectUrlRef.current = url;
            return url;
        }

        if (typeof data.logo === 'string' && data.logo.length) {
            // normalize stored path/URL
            return data.logo.startsWith('http') ? data.logo : `/storage/${data.logo}`;
        }

        return null;
    }, [data.logo]);

    useEffect(() => {
        return () => {
            if (objectUrlRef.current) URL.revokeObjectURL(objectUrlRef.current);
        };
    }, []);

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();

        const fd = new FormData();
        fd.append('name', data.name);
        fd.append('is_active', data.is_active ? '1' : '0');

        // Only append if user selected a NEW file
        if (data.logo instanceof File) {
            fd.append('logo', data.logo);
        }

        if (bank) {
            // UPDATE (keep existing logo if no new file)
            fd.append('_method', 'PUT');
            router.post(route('admin.banks.update', bank.id), fd, {
                onSuccess: () => {
                    onClose();
                    reset();
                }
            });
        } else {
            // CREATE (logo required on create -> enforce via prop below)
            router.post(route('admin.banks.store'), fd, {
                onSuccess: () => {
                    onClose();
                    reset();
                }
            });
        }
    };

    return (
        <Sheet open={isOpen} onOpenChange={(open) => !open && onClose()}>
            <SheetContent side="right" className="w-full sm:max-w-md bg-gray-50">
                <SheetHeader>
                    <SheetTitle>{bank ? 'Edit Bank' : 'Add New Bank'}</SheetTitle>
                    <SheetDescription>
                        {bank ? 'Update the bank details below.' : 'Fill in the details to add a new bank.'}
                    </SheetDescription>
                </SheetHeader>

                <form onSubmit={handleSubmit} className="space-y-6 pb-4 px-4">
                    <div className="space-y-2">
                        <Label htmlFor="name">Bank Name</Label>
                        <Input
                            id="name"
                            value={data.name}
                            onChange={(e) => setData('name', e.target.value)}
                            error={errors.name}
                            required
                        />
                        {errors.name && <p className="text-sm text-red-500">{errors.name}</p>}
                    </div>

                    <div className="space-y-2">
                        <Label htmlFor="logo">Bank Logo</Label>
                        <FileInput
                            id="logo"
                            defaultValue={data.logo}
                            accept="image/*"
                            error={errors.logo}
                            value={data.logo instanceof File ? data.logo : null}
                            onChange={(file) => setData('logo', file)}   // file: File | null
                        />
                        {errors.logo && <p className="text-sm text-red-500">{errors.logo}</p>}
                    </div>

                    <div className="flex items-center space-x-2">
                        <Checkbox
                            id="is_active"
                            checked={data.is_active}
                            onCheckedChange={(checked) => setData('is_active', Boolean(checked))}
                        />
                        <Label htmlFor="is_active" className="text-sm font-medium leading-none">
                            Active
                        </Label>
                    </div>

                    <SheetFooter className="pt-4">
                        <SheetClose asChild>
                            <Button type="button" variant="outline">Cancel</Button>
                        </SheetClose>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Saving...' : 'Save'}
                        </Button>
                    </SheetFooter>
                </form>
            </SheetContent>
        </Sheet>
    );
}
