import { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import { Bank, BankAccount, PageProps, PaginationProps } from '@/types';
import AdminLayout from '@/layouts/admin-layout';
import { Button } from '@/components/ui/button';
import { Plus, Pencil, Trash } from 'lucide-react';
import Swal from 'sweetalert2';
import BankAccountForm from './form';
import { generateInital } from '@/lib/utils';
import EmptyData from '@/components/empty-data';
import Pagination from '@/components/ui/pagination';

type Props = PageProps & {
    bankAccounts: PaginationProps<BankAccount>;
    banks: Bank[];
};

export default function Index({ bankAccounts, banks }: Props) {
    const [isFormOpen, setIsFormOpen] = useState(false);
    const [selectedBankAccount, setSelectedBankAccount] = useState<number | null>(null);

    const handleAddNew = () => {
        setSelectedBankAccount(null);
        setIsFormOpen(true);
    };

    const handleEdit = (id: number) => {
        setSelectedBankAccount(id);
        setIsFormOpen(true);
    };

    const handleCloseForm = () => {
        setIsFormOpen(false);
        setSelectedBankAccount(null);
    };

    const confirmDelete = (id: number, accountNumber: string) => {
        Swal.fire({
            title: 'Hapus Rekening Bank?',
            html: `Apakah Anda yakin ingin menghapus rekening <b>${accountNumber}</b>?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Ya, Hapus',
            cancelButtonText: 'Batal',
            reverseButtons: true
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(route('admin.bank-accounts.destroy', id), {
                    preserveScroll: true,
                    onSuccess: () => {
                        Swal.fire({
                            title: 'Terhapus!',
                            text: 'Rekening bank berhasil dihapus.',
                            icon: 'success',
                            timer: 1500,
                            showConfirmButton: false
                        });
                    },
                    onError: () => {
                        Swal.fire('Error', 'Gagal menghapus rekening bank.', 'error');
                    }
                });
            }
        });
    };

    const currentBankAccount = selectedBankAccount
        ? bankAccounts.data.find(acc => acc.id === selectedBankAccount)
        : null;

    return (
        <AdminLayout>
            <Head title="Rekening Bank" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6 bg-white">
                            <div className="flex justify-between items-center mb-6">
                                <h2 className="text-2xl font-semibold">Daftar Rekening Bank</h2>
                                <Button onClick={handleAddNew}>
                                    <Plus className="h-4 w-4 mr-2" />
                                    Tambah Rekening
                                </Button>
                            </div>

                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                    <tr>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            No.
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            Bank
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            Nomor Rekening
                                        </th>
                                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            Nama Pemilik
                                        </th>
                                        <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                            Aksi
                                        </th>
                                    </tr>
                                    </thead>
                                    <tbody className="bg-white divide-y divide-gray-200">
                                    {bankAccounts.data.map((account, index) => (
                                        <tr key={account.id} className="hover:bg-gray-50">
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                                {index + 1}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap">
                                                <div className="flex items-center space-x-2">
                                                    {account?.bank?.logo ? (
                                                        <img
                                                            className="h-8 w-8 rounded-full"
                                                            src={`/storage/${account.bank.logo}`}
                                                            alt={account.bank.name}
                                                        />
                                                    ) : (
                                                        <div
                                                            className="h-8 w-8 rounded-full bg-gray-200 flex items-center justify-center">
                                                            <span
                                                                className="text-gray-500 text-xs">{generateInital(account?.bank?.name ?? '')}</span>
                                                        </div>
                                                    )}
                                                    <div className="text-sm font-medium text-gray-900">
                                                        {account?.bank?.name}
                                                    </div>
                                                </div>
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                {account.account_number}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                {account.account_name}
                                            </td>
                                            <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                                <button
                                                    onClick={() => handleEdit(account.id)}
                                                    className="text-blue-600 hover:text-blue-900 mr-4"
                                                    title="Edit"
                                                >
                                                    <Pencil className="h-5 w-5" />
                                                </button>
                                                <button
                                                    onClick={() => confirmDelete(account.id, account.account_number)}
                                                    className="text-red-600 hover:text-red-900"
                                                    title="Hapus"
                                                >
                                                    <Trash className="h-5 w-5" />
                                                </button>
                                            </td>
                                        </tr>
                                    ))}
                                    </tbody>
                                </table>
                                {bankAccounts.data.length === 0 && <EmptyData theme="light" />}
                            </div>

                            {/* Pagination */}
                            <Pagination
                                current_page={bankAccounts.current_page}
                                last_page={bankAccounts.last_page}
                                total={bankAccounts.total}
                                per_page={bankAccounts.per_page}
                                links={bankAccounts.links}
                            />
                        </div>
                    </div>
                </div>
            </div>
            <BankAccountForm
                key={selectedBankAccount || 'create'}
                banks={banks}
                bankAccount={currentBankAccount}
                isOpen={isFormOpen}
                onClose={handleCloseForm}
            />
        </AdminLayout>
    );
}
