import { useForm } from '@inertiajs/react';
import { Bank, BankAccount } from '@/types';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Label } from '@/components/ui/label';
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from '@/components/ui/select';
import {
    Sheet,
    SheetContent,
    SheetHeader,
    SheetTitle,
    SheetDescription,
    SheetFooter,
    SheetClose,
} from '@/components/ui/sheet';

interface BankAccountFormProps {
    banks: Bank[];
    bankAccount?: BankAccount | null;
    onClose: () => void;
    isOpen: boolean;
}

export default function BankAccountForm({ banks, bankAccount, onClose, isOpen }: BankAccountFormProps) {
    const { data, setData, post, put, processing, errors, reset } = useForm({
        bank_id: bankAccount?.bank_id ?? '',
        account_number: bankAccount?.account_number ?? '',
        account_name: bankAccount?.account_name ?? '',
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();

        if (bankAccount) {
            put(route('admin.bank-accounts.update', bankAccount.id), {
                onSuccess: () => {
                    onClose();
                    reset();
                },
            });
        } else {
            post(route('admin.bank-accounts.store'), {
                onSuccess: () => {
                    onClose();
                    reset();
                },
            });
        }
    };

    return (
        <Sheet open={isOpen} onOpenChange={(open) => !open && onClose()}>
            <SheetContent side="right" className="w-full sm:max-w-md bg-gray-50">
                <SheetHeader>
                    <SheetTitle>{bankAccount ? 'Edit Rekening Bank' : 'Tambah Rekening Bank'}</SheetTitle>
                    <SheetDescription>
                        {bankAccount ? 'Perbarui detail rekening bank' : 'Isi detail untuk menambahkan rekening bank baru'}
                    </SheetDescription>
                </SheetHeader>

                <form onSubmit={handleSubmit} className="space-y-6 pb-4 px-4">
                    <div className="space-y-2">
                        <Label htmlFor="bank_id">Bank</Label>
                        <Select
                            value={data.bank_id.toString()}
                            onValueChange={(value) => setData('bank_id', parseInt(value))}
                        >
                            <SelectTrigger>
                                <SelectValue placeholder="Pilih Bank" />
                            </SelectTrigger>
                            <SelectContent>
                                {banks.map((bank) => (
                                    <SelectItem key={bank.id} value={bank.id.toString()}>
                                        {bank.name}
                                    </SelectItem>
                                ))}
                            </SelectContent>
                        </Select>
                        {errors.bank_id && <p className="text-sm text-red-500">{errors.bank_id}</p>}
                    </div>

                    <div className="space-y-2">
                        <Label htmlFor="account_number">Nomor Rekening</Label>
                        <Input
                            id="account_number"
                            value={data.account_number}
                            onChange={(e) => setData('account_number', e.target.value)}
                            error={errors.account_number}
                            required
                        />
                    </div>

                    <div className="space-y-2">
                        <Label htmlFor="account_name">Nama Pemilik Rekening</Label>
                        <Input
                            id="account_name"
                            value={data.account_name}
                            onChange={(e) => setData('account_name', e.target.value)}
                            error={errors.account_name}
                            required
                        />
                    </div>

                    <SheetFooter className="pt-4">
                        <SheetClose asChild>
                            <Button type="button" variant="outline">Batal</Button>
                        </SheetClose>
                        <Button type="submit" disabled={processing}>
                            {processing ? 'Menyimpan...' : 'Simpan'}
                        </Button>
                    </SheetFooter>
                </form>
            </SheetContent>
        </Sheet>
    );
}
