interface ApiOptions {
    method?: 'GET' | 'POST' | 'PUT' | 'PATCH' | 'DELETE';
    body?: BodyInit | Record<string, any> | undefined; // allow FormData or JSON
    headers?: Record<string, string>;
}

class ApiClient {
    async request(url: string, options: ApiOptions = {}) {
        const {
            method = 'GET',
            body,
            headers = {},
        } = options;

        const isFormData =
            typeof FormData !== 'undefined' && body instanceof FormData;

        // Build headers: never set Content-Type for FormData
        const baseHeaders: Record<string, string> = {
            'X-Requested-With': 'XMLHttpRequest',
            ...(isFormData ? {} : { 'Content-Type': 'application/json' }),
            ...headers,
        };

        const requestOptions: RequestInit = {
            method,
            headers: baseHeaders,
            credentials: 'same-origin',
        };

        if (body !== undefined) {
            requestOptions.body = isFormData ? (body as BodyInit) : JSON.stringify(body);
        }

        try {
            return fetch(url, requestOptions);
        } catch (error) {
            console.error('API request failed:', error);
            throw error;
        }
    }

    get(url: string, options?: Omit<ApiOptions, 'method' | 'body'>) {
        return this.request(url, { ...options, method: 'GET' });
    }
    post(url: string, options?: Omit<ApiOptions, 'method'>) {
        return this.request(url, { ...options, method: 'POST' });
    }
    put(url: string, options?: Omit<ApiOptions, 'method'>) {
        return this.request(url, { ...options, method: 'PUT' });
    }
    patch(url: string, options?: Omit<ApiOptions, 'method'>) {
        return this.request(url, { ...options, method: 'PATCH' });
    }
    delete(url: string, options?: Omit<ApiOptions, 'method' | 'body'>) {
        return this.request(url, { ...options, method: 'DELETE' });
    }
}

export const apiClient = new ApiClient();

export function useApiClient() {
    return apiClient;
}
