import { PropsWithChildren } from 'react';
import { Link, router } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import {
    LayoutDashboard,
    Users,
    CreditCard,
    LogOut,
    Menu,
    X,
    Building2,
    Newspaper,
    Ticket,
    Wallet,
    Mail,
    MessageSquare,
    Settings,
} from 'lucide-react';
import { useState, useEffect } from 'react';
import { ToastContainer } from 'react-toastify';
import 'react-toastify/dist/ReactToastify.css';

interface AdminLayoutProps {
}

export default function AdminLayout({ children }: PropsWithChildren<AdminLayoutProps>) {
    const [sidebarOpen, setSidebarOpen] = useState(false);
    const [unreadCount, setUnreadCount] = useState(0);
    // CSRF token is now handled automatically by Inertia.js

    const navigation = [
        {
            name: 'Dashboard',
            href: route('admin.dashboard'),
            icon: LayoutDashboard,
            group: null
        },
        {
            name: 'Pengguna',
            href: route('admin.users.index'),
            icon: Users,
            group: 'user_management'
        },
        {
            name: 'Transaksi',
            href: route('admin.transactions.index'),
            icon: CreditCard,
            group: 'transactions'
        },
        {
            name: 'Master Bank',
            href: route('admin.banks.index'),
            icon: Building2,
            group: 'master_data'
        },
        {
            name: 'Rekening Bank',
            href: route('admin.bank-accounts.index'),
            icon: Wallet,
            group: 'master_data'
        },
        {
            name: 'Berita',
            href: route('admin.news.index'),
            icon: Newspaper,
            group: 'content'
        },
        {
            name: 'Tiket Bantuan',
            href: route('admin.tickets.index'),
            icon: Ticket,
            group: 'support'
        },
        {
            name: 'Mesej Sistem',
            href: route('admin.system-messages.index'),
            icon: MessageSquare,
            group: 'settings'
        },
        {
            name: 'Tetapan Sistem',
            href: route('admin.system-settings.index'),
            icon: Settings,
            group: 'settings'
        },
    ];

    // Fetch unread count for admin
    useEffect(() => {
        const fetchUnreadCount = async () => {
            try {
                const response = await fetch(route('admin.tickets.unread-count'), {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                });

                if (response.ok) {
                    const data = await response.json();
                    setUnreadCount(data.unread_count || 0);
                }
            } catch (error) {
                console.error('Failed to fetch unread count:', error);
            }
        };

        fetchUnreadCount();

        // Refresh unread count every 30 seconds
        const interval = setInterval(fetchUnreadCount, 30000);

        return () => clearInterval(interval);
    }, []);

    const handleLogout = () => {
        router.post(route('logout'));
    };

    return (
        <div className="min-h-screen bg-gray-50">
            {/* Mobile sidebar */}
            <div className={`fixed inset-0 z-50 lg:hidden ${sidebarOpen ? 'block' : 'hidden'}`}>
                <div className="fixed inset-0 bg-black/50" onClick={() => setSidebarOpen(false)} />
                <div className="fixed inset-y-0 left-0 flex w-64 flex-col bg-white">
                    <div className="flex h-16 items-center justify-between px-4">
                        <h1 className="text-xl font-bold text-gray-900">Admin Panel</h1>
                        <Button variant="ghost" size="sm" onClick={() => setSidebarOpen(false)}>
                            <X className="w-5 h-5" />
                        </Button>
                    </div>
                    <nav className="flex-1 space-y-1 px-2 py-4">
                        {navigation.map((item) => (
                            <Link
                                key={item.name}
                                href={item.href}
                                className="group flex items-center px-2 py-2 text-sm font-medium rounded-md text-gray-600 hover:bg-gray-50 hover:text-gray-900"
                            >
                                <item.icon className="mr-3 h-5 w-5" />
                                {item.name}
                                {item.name === 'Tiket Bantuan' && unreadCount > 0 && (
                                    <span className="ml-auto text-xs text-white bg-red-500 rounded-full w-5 h-5 flex items-center justify-center font-medium">
                                        {unreadCount > 9 ? '9+' : unreadCount}
                                    </span>
                                )}
                            </Link>
                        ))}
                    </nav>
                    <div className="border-t border-gray-200 p-4">
                        <Button onClick={handleLogout} variant="outline" className="w-full">
                            <LogOut className="w-4 h-4 mr-2" />
                            Log Keluar
                        </Button>
                    </div>
                </div>
            </div>

            {/* Desktop sidebar */}
            <div className="hidden lg:fixed lg:inset-y-0 lg:flex lg:w-64 lg:flex-col">
                <div className="flex flex-col flex-grow bg-white border-r border-gray-200">
                    <div className="flex h-16 items-center px-4">
                        <h1 className="text-xl font-bold text-gray-900">Admin Panel</h1>
                    </div>
                    <nav className="flex-1 space-y-1 px-2 py-4">
                        {navigation.map((item) => (
                            <Link
                                key={item.name}
                                href={item.href}
                                className="group flex items-center px-2 py-2 text-sm font-medium rounded-md text-gray-600 hover:bg-gray-50 hover:text-gray-900"
                            >
                                <item.icon className="mr-3 h-5 w-5" />
                                {item.name}
                                {item.name === 'Tiket Bantuan' && unreadCount > 0 && (
                                    <span className="ml-auto text-xs text-white bg-red-500 rounded-full w-5 h-5 flex items-center justify-center font-medium">
                                        {unreadCount > 9 ? '9+' : unreadCount}
                                    </span>
                                )}
                            </Link>
                        ))}
                    </nav>
                    <div className="border-t border-gray-200 p-4">
                        <Button onClick={handleLogout} variant="outline" className="w-full">
                            <LogOut className="w-4 h-4 mr-2" />
                            Log Keluar
                        </Button>
                    </div>
                </div>
            </div>

            {/* Main content */}
            <div className="lg:pl-64">
                {/* Top bar */}
                <div className="sticky top-0 z-40 flex h-16 shrink-0 items-center justify-between gap-x-4 border-b border-gray-200 bg-white px-4 shadow-sm sm:gap-x-6 sm:px-6 lg:px-8">
                    <Button
                        variant="ghost"
                        size="sm"
                        className="lg:hidden"
                        onClick={() => setSidebarOpen(true)}
                    >
                        <Menu className="w-8 h-8" />
                    </Button>

                    {/* Unread tickets indicator */}
                    <div className="flex items-center gap-4">
                        <Link
                            href={route('admin.tickets.index')}
                            className="relative cursor-pointer rounded-full w-10 h-10 bg-blue-600 hover:bg-blue-700 flex items-center justify-center transition-colors"
                        >
                            {unreadCount > 0 && (
                                <span className="absolute -top-1 -right-1 text-xs text-white bg-red-500 rounded-full w-5 h-5 flex items-center justify-center font-medium">
                                    {unreadCount > 9 ? '9+' : unreadCount}
                                </span>
                            )}
                            <Mail className="w-5 h-5 text-white" />
                        </Link>
                    </div>
                </div>

                {/* Page content */}
                <main className="flex-1">
                    {children}
                </main>
            </div>

            {/* Toast Container */}
            <ToastContainer
                position="top-right"
                autoClose={5000}
                hideProgressBar={false}
                newestOnTop={false}
                closeOnClick
                rtl={false}
                pauseOnFocusLoss={false}
                draggable
                pauseOnHover={false}
                theme="colored"
            />
        </div>
    );
}
