import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle, SheetTrigger } from "@/components/ui/sheet";
import { cn, formatAmount, generateInital } from '@/lib/utils';
import { BanknoteArrowDown, CheckCircle, Loader2 } from 'lucide-react';
import { router, usePage } from '@inertiajs/react';
import { BankAccount } from "@/types";
import { useApiClient } from '@/lib/api';
import PinVerification from './PinVerification';

type WithdrawalProps = {
    /** Optional custom trigger element (e.g., <Button/>, <a/>, etc.) */
    trigger?: React.ReactElement;
};

const Withdrawal = ({ trigger }: WithdrawalProps) => {
    const [showModal, setShowModal] = useState(false);
    const [selectedBank, setSelectedBank] = useState<BankAccount | null>(null);
    const [amount, setAmount] = useState("0");
    const [step, setStep] = useState<"amount" | "bank" | "pin" | "loading" | "success">("amount");
    const [isLoading, setIsLoading] = useState(false);
    const [bankAccounts, setBankAccounts] = useState<BankAccount[]>([]);
    const [userBalance, setUserBalance] = useState(0);
    const apiClient = useApiClient();
    const {csrf_token} = usePage().props;

    // Fetch banks and user balance from backend
    useEffect(() => {
        const fetchData = async () => {
            try {
                const [banksResponse, balanceResponse] = await Promise.all([
                    apiClient.get("/api/bank-account/user"),
                    apiClient.get("/api/user/balance")
                ]);

                if (banksResponse.ok) {
                    const banksData = await banksResponse.json();
                    setBankAccounts(banksData);
                }

                if (balanceResponse.ok) {
                    const balanceData = await balanceResponse.json();
                    setUserBalance(balanceData.balance || 0);
                }
            } catch (error) {
                console.error("Error fetching data:", error);
            }
        };

        fetchData();
    }, [apiClient]);

    const handleSubmit = async () => {
        if (!selectedBank || !amount) return;

        setIsLoading(true);
        setStep("loading");

        try {
            const formData = new FormData();
            if (csrf_token) formData.append('_token', csrf_token.toString());
            formData.append("amount", amount);
            formData.append("bank_account_id", selectedBank.id.toString());

            const response = await apiClient.post(route("transactions.withdrawal"), {
                body: formData,
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    setStep("success");
                    // Refresh the page to show new transaction
                    setTimeout(() => {
                        router.reload();
                    }, 2000);
                } else {
                    console.error("Withdrawal failed:", data.message);
                    setStep("bank");
                }
            } else {
                console.error("Withdrawal request failed");
                setStep("bank");
            }
        } catch (error) {
            console.error("Error submitting withdrawal:", error);
            setStep("bank");
        } finally {
            setIsLoading(false);
        }
    };

    const defaultTrigger = (
        <button
            className="py-4 w-full font-bold text-base border-2 border-primary hover:border-primary-foreground transition-all rounded-full flex space-x-2 items-center justify-center bg-primary-foreground hover:bg-primary text-primary hover:text-primary-foreground"
        >
            <BanknoteArrowDown className="w-6 h-6" />
            <span>Withdrawal Baki</span>
        </button>
    );

    return (
        <Sheet open={showModal} onOpenChange={setShowModal}>
            <SheetTrigger asChild>{trigger ?? defaultTrigger}</SheetTrigger>
            <SheetContent side="bottom" className="max-w-sm mx-auto rounded-t-2xl overflow-y-auto transition-all">
                <SheetHeader className="text-left pb-4">
                    <SheetTitle className="text-lg font-bold text-white">
                        {step === "amount" && "Withdrawal Baki"}
                        {step === "bank" && "Pilih Bank"}
                        {step === "pin" && "Sahkan PIN Transaksi"}
                        {step === "loading" && "Memproses Permohonan"}
                        {step === "success" && "Berjaya Dihantar"}
                    </SheetTitle>
                    <SheetDescription className="text-sm text-muted">
                        {step === "amount" && "Masukkan jumlah yang ingin dikeluarkan"}
                        {step === "bank" && "Pilih bank untuk pemindahan"}
                        {step === "pin" && "Masukkan 6-digit PIN transaksi untuk mengesahkan permohonan"}
                        {step === "loading" && "Sila tunggu sebentar sementara kami memproses permohonan anda..."}
                        {step === "success" && "Permohonan anda telah dihantar untuk pemprosesan"}
                    </SheetDescription>
                </SheetHeader>

                <div className="px-4 space-y-6 pb-6">
                    {/* Step 1: Amount Input */}
                    {step === "amount" && (
                        <>
                            {/* Current Balance Display */}
                            <div className="text-center p-4 bg-primary-foreground/20 rounded-xl">
                                <div className="text-sm text-muted mb-1">Baki Semasa</div>
                                <div className="text-2xl font-bold text-white">
                                    {formatAmount(userBalance)}
                                </div>
                            </div>

                            {/* Amount Input */}
                            <div className="space-y-4">
                                <div>
                                    <label className="text-sm font-medium text-white mb-2 block">Jumlah Withdrawal</label>
                                    <div className="relative">
                                        <span className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground font-bold">
                                            RM
                                        </span>
                                        <Input
                                            type="text"
                                            placeholder="0.00"
                                            value={
                                                amount ? parseFloat(amount.replace(/\./g, "")).toLocaleString("ms-MY").replace(/,/g, ".") : ""
                                            }
                                            onChange={(e) => {
                                                const value = e.target.value.replace(/\./g, "");
                                                if (/^\d*$/.test(value)) {
                                                    setAmount(value);
                                                }
                                            }}
                                            className="pl-12 text-lg font-medium h-12"
                                        />
                                    </div>
                                    {parseFloat(amount) > userBalance && (
                                        <p className="text-red-400 text-sm mt-1">
                                            Jumlah melebihi baki semasa
                                        </p>
                                    )}
                                </div>

                                {/* Quick Amount Buttons */}
                                <div className="grid grid-cols-3 gap-3">
                                    {[50, 100, 200].map((quickAmount) => (
                                        <button
                                            key={quickAmount}
                                            onClick={() => setAmount(quickAmount.toString())}
                                            disabled={quickAmount > userBalance}
                                            className={cn(
                                                "p-3 cursor-pointer rounded-lg transition-colors text-sm font-medium",
                                                quickAmount > userBalance
                                                    ? "bg-gray-400 text-gray-600 cursor-not-allowed"
                                                    : "bg-foreground/40 text-background hover:bg-muted/50"
                                            )}
                                        >
                                            RM {quickAmount}
                                        </button>
                                    ))}
                                </div>
                            </div>

                            {/* Submit Button */}
                            <Button
                                onClick={() => amount && setStep("bank")}
                                disabled={!amount || parseFloat(amount) <= 0 || parseFloat(amount) > userBalance}
                                className="w-full bg-primary-foreground hover:bg-primary-foreground/80 text-primary font-bold"
                            >
                                Teruskan
                            </Button>
                        </>
                    )}

                    {/* Step 2: Bank Selection & Receipt Upload */}
                    {step === "bank" && (
                        <>
                            {/* Amount Summary */}
                            <div className="text-center p-4 bg-primary-foreground/20 rounded-xl">
                                <div className="text-sm text-muted mb-1">Jumlah Withdrawal</div>
                                <div className="text-2xl font-bold text-white">
                                    RM {amount ? parseFloat(amount.replace(/\./g, "")).toLocaleString("ms-MY").replace(/,/g, ".") : "0"}
                                </div>
                            </div>

                            {/* Bank Selection */}
                            <div>
                                <h4 className="font-medium mb-3 text-background">Pilih Bank untuk Pemindahan</h4>
                                <div className="space-y-2 max-h-48 overflow-y-auto">
                                    {bankAccounts.map((account) => (
                                        <button
                                            key={account.id}
                                            onClick={() => setSelectedBank(account)}
                                            className={`w-full flex items-center gap-3 p-4 rounded-lg transition-all ${
                                                selectedBank?.id === account.id ? "bg-foreground" : "bg-foreground/40 hover:bg-foreground/80"
                                            }`}
                                        >
                                            {account?.bank?.logo ? (
                                                <img
                                                    className="h-10 w-10 rounded-full"
                                                    src={`/storage/${account.bank.logo}`}
                                                    alt={account.bank.name}
                                                />
                                            ) : (
                                                <div className="h-10 w-10 rounded-full bg-gray-200 flex items-center justify-center">
                                                    <span className="text-gray-500 text-xs">{generateInital(account?.bank?.name ?? "")}</span>
                                                </div>
                                            )}
                                            <div className="flex-1 text-left">
                                                <div className="font-medium text-background">{account?.bank?.name}</div>
                                                <div className="text-sm text-muted/50">{account.account_number}</div>
                                                <div className="text-sm text-muted/50">{account.account_name}</div>
                                            </div>
                                            {selectedBank?.id === account.id && <CheckCircle className="w-6 h-6 text-primary-foreground" />}
                                        </button>
                                    ))}
                                </div>
                            </div>

                            {/* Action Buttons */}
                            <div className="flex gap-3">
                                <Button
                                    variant="outline"
                                    onClick={() => setStep("amount")}
                                    className="flex-1 font-bold border border-primary-foreground bg-primary hover:bg-primary-foreground text-primary-foreground hover:text-primary"
                                >
                                    Kembali
                                </Button>
                                <Button
                                    onClick={() => setStep("pin")}
                                    disabled={!selectedBank || isLoading}
                                    className="flex-1 bg-primary-foreground hover:bg-primary-foreground/80 text-primary font-bold"
                                >
                                    {isLoading ? <Loader2 className="w-4 h-4 animate-spin" /> : "Hantar Permohonan"}
                                </Button>
                            </div>
                        </>
                    )}

                    {/* Step 3: PIN Verification */}
                    {step === "pin" && (
                        <>
                            <div className="text-center p-4 bg-primary-foreground/20 rounded-xl mb-6">
                                <div className="text-sm text-muted mb-1">Ringkasan Permohonan</div>
                                <div className="text-2xl font-bold text-white">
                                    RM {amount ? parseFloat(amount.replace(/\./g, '')).toLocaleString('ms-MY').replace(/,/g, '.') : '0'}
                                </div>
                                <div className="text-sm text-muted mt-2">
                                    {selectedBank?.bank?.name} - {selectedBank?.account_number}
                                </div>
                            </div>

                            <div className="text-center mb-6">
                                <p className="text-muted mb-4">
                                    Sila sahkan PIN transaksi anda untuk meneruskan permohonan withdrawal
                                </p>
                            </div>

                            <PinVerification
                                isOpen={step === "pin"}
                                onClose={() => setStep("bank")}
                                onVerified={() => {
                                    // After PIN verification, automatically submit the transaction
                                    handleSubmit();
                                }}
                                title="Sahkan PIN untuk Withdrawal"
                                description="Masukkan 6-digit PIN transaksi untuk mengesahkan permohonan withdrawal"
                            />
                        </>
                    )}

                    {/* Step 4: Loading */}
                    {step === "loading" && (
                        <>
                            <div className="text-center">
                                <div className="w-24 h-24 bg-primary-foreground/20 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <Loader2 className="w-12 h-12 text-primary-foreground animate-spin" />
                                </div>
                                <h3 className="text-xl font-bold mb-2 text-primary-foreground">Memproses Permohonan</h3>
                                <p className="text-muted mb-6">Sila tunggu sebentar sementara kami memproses permohonan anda...</p>
                            </div>

                            {/* Transaction Summary */}
                            <div className="space-y-4 bg-foreground/20 rounded-lg p-4">
                                <div className="flex justify-between">
                                    <span className="text-muted">Jumlah</span>
                                    <span className="font-bold text-primary-foreground">RM {parseFloat(amount).toFixed(2)}</span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-muted">Bank</span>
                                    <span className="font-medium text-primary-foreground">{selectedBank?.bank?.name}</span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-muted">Status</span>
                                    <span className="text-yellow-500 font-medium">Sedang Diproses...</span>
                                </div>
                            </div>
                        </>
                    )}

                    {/* Step 5: Success */}
                    {step === "success" && (
                        <>
                            <div className="text-center">
                                <div className="w-24 h-24 bg-green-500/10 rounded-full flex items-center justify-center mx-auto mb-4">
                                    <CheckCircle className="w-12 h-12 text-green-500" />
                                </div>
                                <h3 className="text-xl font-bold mb-2 text-primary-foreground">Berjaya Dihantar!</h3>
                                <p className="text-muted mb-6">
                                    Permohonan withdrawal anda telah dihantar. Kami akan memproses segera.
                                </p>
                            </div>

                            {/* Transaction Summary */}
                            <div className="space-y-4 bg-foreground/20 rounded-lg p-4">
                                <div className="flex justify-between">
                                    <span className="text-muted">Jumlah</span>
                                    <span className="font-bold text-primary-foreground">RM {parseFloat(amount).toFixed(2)}</span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-muted">Bank</span>
                                    <span className="font-medium text-primary-foreground">{selectedBank?.bank?.name}</span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-muted">Status</span>
                                    <span className="text-yellow-500 font-medium">Dalam Pemprosesan</span>
                                </div>
                            </div>

                            <Button
                                onClick={() => {
                                    setAmount("0");
                                    setSelectedBank(null);
                                    setStep("amount");
                                    setShowModal(false);
                                }}
                                className="cursor-pointer w-full bg-primary-foreground hover:bg-primary text-primary hover:text-primary-foreground border border-primary-foreground hover:border-primary-foreground font-bold"
                            >
                                Selesai
                            </Button>
                        </>
                    )}
                </div>
            </SheetContent>
        </Sheet>
    );
};

export default Withdrawal;
