import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from '@/components/ui/sheet';
import { Button } from '@/components/ui/button';
import { ChevronLeft, ChevronRight } from 'lucide-react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import dayjs from 'dayjs';
import { useEffect, useState } from 'react';
import { TransactionProps } from '@/types';
import TransactionItem from '@/components/user/transactions/TransactionItem';
import TransactionDetail from '@/components/user/transactions/TransactionDetail';
import { useApiClient } from '@/lib/api';

interface PageProps {
    isOpen: boolean;
    onClose: () => void;
}

const TransactionList = ({ isOpen, onClose }: PageProps) => {
    const [transactions, setTransactions] = useState<TransactionProps[]>([])
    const [selectedTransaction, setSelectedTransaction] = useState<TransactionProps | null>(null);
    const [isLoading, setIsLoading] = useState(false);
    const [currentMonth, setCurrentMonth] = useState(dayjs());
    const apiClient = useApiClient();

    const months = [
        { value: '0', label: 'Januari' },
        { value: '1', label: 'Februari' },
        { value: '2', label: 'Mac' },
        { value: '3', label: 'April' },
        { value: '4', label: 'Mei' },
        { value: '5', label: 'Jun' },
        { value: '6', label: 'Julai' },
        { value: '7', label: 'Ogos' },
        { value: '8', label: 'September' },
        { value: '9', label: 'Oktober' },
        { value: '10', label: 'November' },
        { value: '11', label: 'Disember' }
    ];

    const years = Array.from({ length: 5 }, (_, i) => dayjs().year() - 2 + i);

    const fetchTransactions = async (month: dayjs.Dayjs) => {
        setIsLoading(true);
        try {
            const response = await apiClient.get(route('transactions.filter') + `?month=${month.month()}&year=${month.year()}`);

            if (response.ok) {
                const data = await response.json();
                setTransactions(data.transactions)
            }
        } catch (error) {
            console.error('Error fetching transactions:', error);
        } finally {
            setIsLoading(false);
        }
    };

    const goToPreviousMonth = () => {
        const newMonth = currentMonth.subtract(1, 'month');
        if (!newMonth.isSame(currentMonth, 'month')) {
            setCurrentMonth(newMonth);
            fetchTransactions(newMonth);
        }
    };

    const goToNextMonth = () => {
        const newMonth = currentMonth.add(1, 'month');
        if (!newMonth.isSame(currentMonth, 'month')) {
            setCurrentMonth(newMonth);
            fetchTransactions(newMonth);
        }
    };

    const handleMonthChange = (monthValue: string) => {
        const monthIndex = parseInt(monthValue);
        const newMonth = dayjs().month(monthIndex).year(currentMonth.year());
        if (!newMonth.isSame(currentMonth, 'month')) {
            setCurrentMonth(newMonth);
            fetchTransactions(newMonth);
        }
    };

    const handleYearChange = (yearValue: string) => {
        const newMonth = currentMonth.year(parseInt(yearValue));
        if (!newMonth.isSame(currentMonth, 'year')) {
            setCurrentMonth(newMonth);
            fetchTransactions(newMonth);
        }
    };

    useEffect(() => {
        if (isOpen && currentMonth) fetchTransactions(currentMonth)
    },[isOpen, currentMonth])

    return (
        <>
            <Sheet open={isOpen} onOpenChange={onClose}>
                <SheetContent side="bottom" className="h-[100vh] max-w-sm mx-auto rounded-t-2xl">
                    <SheetHeader className="text-left pb-0">
                        <SheetTitle className="text-lg font-bold text-white">Sejarah Transaksi</SheetTitle>
                        <SheetDescription className="text-sm text-muted">
                            Lihat transaksi mengikut bulan
                        </SheetDescription>
                    </SheetHeader>

                    {/* Month Navigation */}
                    <div className="flex items-center justify-between px-4">
                        <Button
                            variant="ghost"
                            size="sm"
                            onClick={goToPreviousMonth}
                            className="text-white hover:bg-white/20"
                            disabled={isLoading}
                        >
                            <ChevronLeft className="w-4 h-4" />
                        </Button>

                        <div className="flex items-center gap-2">
                            <Select value={currentMonth.month().toString()} onValueChange={handleMonthChange}>
                                <SelectTrigger className="w-32 bg-white/10 border-white/20 text-white">
                                    <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                    {months.map((month) => (
                                        <SelectItem key={month.value} value={month.value}>
                                            {month.label}
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>

                            <Select value={currentMonth.year().toString()} onValueChange={handleYearChange}>
                                <SelectTrigger className="w-24 bg-white/10 border-white/20 text-white">
                                    <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                    {years.map((year) => (
                                        <SelectItem key={year} value={year.toString()}>
                                            {year}
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>

                        <Button
                            variant="ghost"
                            size="sm"
                            onClick={goToNextMonth}
                            className="text-white hover:bg-white/20"
                            disabled={isLoading}
                        >
                            <ChevronRight className="w-4 h-4" />
                        </Button>
                    </div>

                    {/* Monthly Transactions */}
                    <div className="flex-1 overflow-y-auto px-4 space-y-3">
                        {isLoading ? (
                            <div className="text-center py-8">
                                <div className="text-muted">Memuatkan transaksi...</div>
                            </div>
                        ) : transactions.length > 0 ? (
                            transactions.map((transaction) => (
                                <TransactionItem
                                    key={transaction.id}
                                    data={transaction}
                                    onShowDetail={setSelectedTransaction}
                                />
                            ))
                        ) : (
                            <div className="text-center py-8">
                                <div className="text-muted">Tiada transaksi untuk bulan ini</div>
                            </div>
                        )}
                    </div>

                    {/* Close Button */}
                    <div className="p-4 border-t border-white/20">
                        <Button
                            variant="outline"
                            className="w-full"
                            onClick={onClose}
                        >
                            Tutup
                        </Button>
                    </div>
                </SheetContent>
            </Sheet>

            {/* Transaction Detail Sheet */}
            <TransactionDetail
                data={selectedTransaction}
                isOpen={!!selectedTransaction}
                onClose={() => {
                    setSelectedTransaction(null);
                }} />
        </>
    );
};

export default TransactionList;
