import { cn, formatAmount, formatDate } from '@/lib/utils';
import { TransactionProps } from '@/types';
import { Ban, Clock, Minus, Plus } from 'lucide-react';

type Props = {
    data: TransactionProps
    onShowDetail: (data: TransactionProps) => void;
}

export const getStatusVariant = (status: string) => {
    switch (status) {
        case 'completed':
            return 'success';
        case 'processing':
            return 'warning';
        case 'failed':
            return 'destructive';
        default:
            return 'warning';
    }
};

export const getTransactionStatusLabel = (status: string) => {
    switch (status) {
        case 'completed':
            return 'Selesai';
        case 'processing':
            return 'Dalam Proses';
        case 'failed':
            return 'Ditolak';
        default:
            return 'Menunggu';
    }
};

export const getTransactionTypeColor = (type: string) => {
    switch (type) {
        case 'Penambahan System':
        case 'Tambah Baki':
            return 'bg-green-800/30';
        case 'Pengurangan System':
        case 'Pengeluaran':
            return 'bg-yellow-700/50';
        case 'failed':
            return 'bg-destructive/60';
        default:
            return 'bg-yellow-700/50';
    }
};

export const getTransactionStatusColor = (status: string) => {
    switch (status) {
        case 'completed':
            return 'bg-green-800/30';
        case 'processing':
            return 'bg-yellow-700/50';
        case 'failed':
            return 'bg-destructive/60';
        default:
            return 'bg-yellow-700/50';
    }
};

export const getTransactionTypeTextColor = (type: string) => {
    switch (type) {
        case 'Penambahan System':
        case 'Tambah Baki':
            return 'text-green-500';
        case 'Pengurangan System':
        case 'Pengeluaran':
            return 'text-yellow-500';
        case 'failed':
            return 'text-destructive';
        default:
            return 'text-yellow-500';
    }
};

export const getTransactionStatusTextColor = (status: string) => {
    switch (status) {
        case 'completed':
            return 'text-green-500';
        case 'processing':
            return 'text-yellow-500';
        case 'failed':
            return 'text-destructive';
        default:
            return 'text-yellow-500';
    }
};

export const getTransactionTypeIcon = (type: string) => {
    switch (type) {
        case 'Penambahan System':
        case 'Tambah Baki':
            return <Plus className="w-4 h-4 text-green-500" />;
        case 'Pengurangan System':
        case 'Pengeluaran':
            return <Minus className="w-4 h-4 text-yellow-500" />;
        default:
            return <Clock className="w-4 h-4 text-yellow-500" />;
    }
};

export const getTransactionStatusIcon = (status: string) => {
    switch (status) {
        case 'completed':
            return <Plus className="w-4 h-4 text-green-500" />;
        case 'processing':
            return <Clock className="w-4 h-4 text-yellow-500" />;
        case 'failed':
            return <Ban className="w-4 h-4 text-red-500" />;
        default:
            return <Clock className="w-4 h-4 text-yellow-500" />;
    }
};

const TransactionItem = ({data, onShowDetail}: Props) => {
    return (
        <button
            onClick={() => {
                onShowDetail(data)
            }}
            className="w-full flex items-center justify-between p-4 bg-foreground/40 hover:bg-foreground/60 cursor-pointer rounded-lg transition-colors text-left"
        >
            <div className="flex items-center gap-3">
                <div className={cn(
                    `w-8 h-8 flex items-center justify-center rounded-md`,
                    getTransactionTypeColor(data.type)
                )}>
                    {getTransactionTypeIcon(data.type)}
                </div>
                <div>
                    <div className="font-medium text-sm text-white">{data.type}</div>
                    <div className="text-xs text-muted/70">
                        {formatDate(data.created_at)}
                    </div>
                </div>
            </div>
            <div className="text-right">
                <div className={cn(
                    `font-medium`,
                    getTransactionTypeTextColor(data.type)
                )}>
                    {formatAmount(data.amount)}
                </div>
                <div className={cn(
                    `text-xs capitalize`,
                    data.status === 'failed' && 'text-red-500',
                    data.status === 'processing' && 'text-yellow-500',
                    data.status === 'completed' && 'text-green-500',
                    data.status === 'pending' && 'text-yellow-500'
                )}>
                    {getTransactionStatusLabel(data.status)}
                </div>
            </div>
        </button>
    )
}

export default TransactionItem;
