import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from '@/components/ui/sheet';
import dayjs from 'dayjs';
import { cn, formatAmount } from '@/lib/utils';
import { FileText } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { TransactionProps } from '@/types';
import { getTransactionStatusLabel } from '@/components/user/transactions/TransactionItem';

type Props = {
    isOpen: boolean;
    data: TransactionProps | null;
    onClose: () => void;
}

const TransactionDetail = ({isOpen, data, onClose}: Props) => {
    return (
        <Sheet open={isOpen} onOpenChange={onClose}>
            <SheetContent side="bottom" className="max-w-sm mx-auto rounded-t-2xl">
                <SheetHeader className="text-left">
                    <SheetTitle className="text-lg font-bold text-white">Butiran Transaksi</SheetTitle>
                    <SheetDescription className="text-sm text-muted">
                        Maklumat lengkap transaksi anda
                    </SheetDescription>
                </SheetHeader>

                {data && (
                    <div className="space-y-6 pb-6 px-4 overflow-y-auto max-h-[70vh]">
                        {/* Amount Display */}
                        <div className="text-center p-6 bg-primary/10 rounded-xl">
                            <div className={cn(
                                'text-4xl font-bold',
                                Number(data.amount) > 0 && 'text-green-500',
                                Number(data.amount) < 0 && 'text-yellow-500'
                            )}>
                                {formatAmount(data.amount)}
                            </div>
                        </div>

                        {/* Transaction Details */}
                        <div className="space-y-4">
                            <div className="flex justify-between py-3 border-b border-border">
                                <span className="text-muted">Rujukan</span>
                                <span className="font-medium text-white">{data.reference}</span>
                            </div>
                            <div className="flex justify-between py-3 border-b border-border">
                                <span className="text-muted">Jenis</span>
                                <span className="font-medium text-white">{data.type}</span>
                            </div>
                            <div className="flex justify-between py-3 border-b border-border">
                                <span className="text-muted">Tarikh</span>
                                <span className="font-medium text-white">{dayjs(data.created_at).format('DD-MMM-YYYY')}</span>
                            </div>
                            <div className="flex justify-between py-3 border-b border-border">
                                <span className="text-muted">Masa</span>
                                <span className="font-medium text-white">{dayjs(data.created_at).format('HH:mm:ss')}</span>
                            </div>
                            <div className="flex justify-between py-3 border-b border-border">
                                <span className="text-muted">Penerima</span>
                                <span className="font-medium text-white">{data.recipient}</span>
                            </div>
                            <div className="flex justify-between py-3 border-b border-border">
                                <span className="text-muted">Status</span>
                                <span className={cn(
                                    `font-medium`,
                                    data.status === 'failed' && 'text-red-500',
                                    data.status === 'processing' && 'text-yellow-500',
                                    data.status === 'completed' && 'text-green-500',
                                    data.status === 'pending' && 'text-yellow-500'
                                )}>
                                        {getTransactionStatusLabel(data.status)}
                                    </span>
                            </div>
                            {data.notes && (
                                <div className="flex justify-between py-3 border-b border-border">
                                    <span className="text-muted">Nota</span>
                                    <span className="font-medium text-white">{data.notes}</span>
                                </div>
                            )}
                            <div className="py-3">
                                <div className="text-muted mb-2">Bukti Transaksi</div>
                                {data.evidence ? (
                                    <div
                                        className="relative cursor-pointer"
                                        onClick={() => window.open(`/storage/${data.evidence}`, '_blank')}
                                    >
                                        <img
                                            src={`/storage/${data.evidence}`}
                                            alt="Bukti Transaksi"
                                            className="w-full h-48 object-cover rounded-lg border border-border"
                                        />
                                    </div>
                                ) : (
                                    <div className="text-center py-8 text-muted-foreground">
                                        <FileText className="w-12 h-12 mx-auto mb-2 opacity-50" />
                                        <p className="text-sm">Tiada bukti transaksi</p>
                                    </div>
                                )}
                            </div>
                        </div>

                        {/* Action Buttons */}
                        <Button variant='primary' className="w-full" onClick={onClose}>
                            Tutup
                        </Button>
                    </div>
                )}
            </SheetContent>
        </Sheet>
    )
}

export default TransactionDetail;
