import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import {
    Sheet,
    SheetContent,
    SheetDescription,
    SheetHeader,
    SheetTitle,
    SheetTrigger
} from '@/components/ui/sheet';
import { cn, generateInital } from '@/lib/utils';
import { BanknoteArrowUp, Camera, CheckCircle, X, Loader2, Copy } from 'lucide-react';
import { useState, useEffect } from 'react';
import { router, usePage } from '@inertiajs/react';
import { BankAccount } from '@/types';
import { useApiClient } from '@/lib/api';
import PinVerification from './PinVerification';

type TopupProps = {
    /** Optional custom trigger element (e.g., <Button/>, <a/>, etc.) */
    trigger?: React.ReactElement;
};

// utils/copy-to-clipboard.ts
import { toast } from 'react-toastify';

async function copyToClipboard(text: string) {
    try {
        if (!text) return;
        if (navigator.clipboard && window.isSecureContext) {
            await navigator.clipboard.writeText(text);
        } else {
            const ta = document.createElement('textarea');
            ta.value = text;
            ta.setAttribute('readonly', '');
            ta.style.position = 'fixed';
            ta.style.left = '-9999px';
            document.body.appendChild(ta);
            ta.select();
            document.execCommand('copy');
            document.body.removeChild(ta);
        }
        toast.success('Copied to clipboard');
    } catch {
        toast.error('Copy failed');
    }
}


const Topup = ({ trigger }: TopupProps) => {
    const [showModal, setShowModal] = useState(false);
    const [selectedBank, setSelectedBank] = useState<BankAccount | null>(null);
    const [amount, setAmount] = useState('0');
    const [step, setStep] = useState<'amount' | 'bank' | 'pin' | 'loading' | 'success'>('amount');
    const [evidence, setEvidence] = useState<File | null>(null);
    const [isLoading, setIsLoading] = useState(false);
    const [bankAccounts, setBankAccounts] = useState<BankAccount[]>([]);
    const [error, setError] = useState<string | null>(null);
    const apiClient = useApiClient();
    const {csrf_token} = usePage().props;

    // Fetch banks from backend
    useEffect(() => {
        const fetchBanks = async () => {
            try {
                const response = await apiClient.get('/api/bank-account/admin');
                if (response.ok) {
                    const data = await response.json();
                    setBankAccounts(data);
                }
            } catch (error) {
                console.error('Error fetching banks:', error);
            }
        };

        fetchBanks();
    }, [apiClient]);

    const handleSubmit = async () => {
        if (!selectedBank || !evidence || !amount) return;

        setIsLoading(true);
        setStep('loading');
        setError(null); // Clear any previous errors

        try {
            const formData = new FormData();
            if (csrf_token) formData.append('_token', csrf_token.toString());
            formData.append('amount', amount);
            formData.append('bank_account_id', selectedBank.id.toString());
            formData.append('evidence', evidence);

            const response = await apiClient.post(route('transactions.topup'), {
                body: formData
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    setStep('success');
                    // Refresh the page to show new transaction
                    setTimeout(() => {
                        router.reload();
                    }, 2000);
                } else {
                    console.error('Topup failed:', data.message);
                    setError(data.message || 'Permohonan gagal. Sila cuba lagi.');
                    setStep('bank');
                }
            } else {
                // Handle validation errors and other HTTP errors
                const errorData = await response.json();
                if (errorData.errors && errorData.errors.evidence) {
                    // Handle evidence validation error specifically
                    setError(errorData.errors.evidence[0]);
                } else if (errorData.error) {
                    setError(errorData.error);
                } else if (errorData.message) {
                    setError(errorData.message);
                } else {
                    setError('Permohonan gagal. Sila cuba lagi.');
                }
                setStep('bank');
            }
        } catch (error) {
            console.error('Error submitting topup:', error);
            setError('Ralat sistem. Sila cuba lagi.');
            setStep('bank');
        } finally {
            setIsLoading(false);
        }
    };

    const defaultTrigger = (
        <button
            className="py-4 w-full font-bold text-base border-2 border-primary hover:border-primary-foreground transition-all rounded-full flex space-x-2 items-center justify-center bg-primary-foreground hover:bg-primary text-primary hover:text-primary-foreground"
        >
            <BanknoteArrowUp className="w-6 h-6" />
            <span>Tambah Baki</span>
        </button>
    );

    return (
        <Sheet open={showModal} onOpenChange={(open) => {
            setShowModal(open);
            if (!open) {
                // Reset form when modal is closed
                setAmount('0');
                setEvidence(null);
                setSelectedBank(null);
                setStep('amount');
                setError(null);
            }
        }}>
            <SheetTrigger asChild>{trigger ?? defaultTrigger}</SheetTrigger>
            <SheetContent side="bottom" className="max-w-sm mx-auto rounded-t-2xl overflow-y-auto transition-all">
                <SheetHeader className="text-left pb-4">
                    <SheetTitle className="text-lg font-bold text-white">
                        {step === 'amount' && 'Tambah Baki'}
                        {step === 'bank' && 'Pilih Bank & Muat Naik Resit'}
                        {step === 'pin' && 'Sahkan PIN Transaksi'}
                        {step === 'loading' && 'Memproses Permohonan'}
                        {step === 'success' && 'Berjaya Dihantar'}
                    </SheetTitle>
                    <SheetDescription className="text-sm text-muted">
                        {step === 'amount' && 'Masukkan jumlah yang ingin ditambah'}
                        {step === 'bank' && 'Pilih bank untuk pemindahan dan muat naik bukti'}
                        {step === 'pin' && 'Masukkan 6-digit PIN transaksi untuk mengesahkan permohonan'}
                        {step === 'loading' && 'Sila tunggu sebentar sementara kami memproses permohonan anda...'}
                        {step === 'success' && 'Permohonan anda telah dihantar untuk pemprosesan'}
                    </SheetDescription>
                </SheetHeader>

                <div className="px-4 space-y-6 pb-6">
                    {/* Step 1: Amount Input */}
                    {step === 'amount' && (
                        <>
                            {/* Amount Input */}
                            <div className="space-y-4">
                                <div>
                                    <label className="text-sm font-medium text-white mb-2 block">Jumlah Tambah
                                        Baki</label>
                                    <div className="relative">
                    <span
                        className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground font-bold">
                      RM
                    </span>
                                        <Input
                                            type="text"
                                            placeholder="0.00"
                                            value={
                                                amount ? parseFloat(amount.replace(/\./g, '')).toLocaleString('ms-MY').replace(/,/g, '.') : ''
                                            }
                                            onChange={(e) => {
                                                const value = e.target.value.replace(/\./g, '');
                                                if (/^\d*$/.test(value)) {
                                                    setAmount(value);
                                                }
                                            }}
                                            className="pl-12 text-lg font-medium h-12"
                                        />
                                    </div>
                                </div>

                                {/* Quick Amount Buttons */}
                                <div className="grid grid-cols-3 gap-3">
                                    {[500, 1000, 2500].map((quickAmount) => (
                                        <button
                                            key={quickAmount}
                                            onClick={() => setAmount(quickAmount.toString())}
                                            className="p-3 cursor-pointer bg-foreground/40 text-background hover:bg-muted/50 rounded-lg transition-colors text-sm font-medium"
                                        >
                                            RM {quickAmount}
                                        </button>
                                    ))}
                                </div>
                            </div>

                            {/* Submit Button */}
                            <Button
                                onClick={() => {
                                    setError(null); // Clear any previous errors
                                    amount && setStep('bank');
                                }}
                                disabled={!amount || parseFloat(amount) <= 0}
                                className="w-full bg-primary-foreground hover:bg-primary-foreground/80 text-primary font-bold"
                            >
                                Teruskan
                            </Button>
                        </>
                    )}

                    {/* Step 2: Bank Selection & Receipt Upload */}
                    {step === 'bank' && (
                        <>
                            {/* Amount Summary */}
                            <div className="text-center p-4 bg-primary-foreground/20 rounded-xl">
                                <div className="text-sm text-muted mb-1">Jumlah Tambah Baki</div>
                                <div className="text-2xl font-bold text-white">
                                    RM {amount ? parseFloat(amount.replace(/\./g, '')).toLocaleString('ms-MY').replace(/,/g, '.') : '0'}
                                </div>
                            </div>

                            {/* Bank Selection */}
                            <div>
                                <h4 className="font-medium mb-3 text-background">Pilih Bank untuk Pemindahan</h4>
                                <div className="space-y-2 max-h-48 overflow-y-auto">
                                    {bankAccounts.map((account) => (
                                        <button
                                            key={account.id}
                                            onClick={() => setSelectedBank(account)}
                                            className={`w-full flex items-center gap-3 p-4 rounded-lg border-2 transition-all ${
                                                selectedBank?.id === account.id ? 'bg-foreground border-primary-foreground' : 'border-foreground/40 bg-foreground/40 hover:bg-foreground/80'
                                            }`}
                                        >
                                            {account?.bank?.logo ? (
                                                <img
                                                    className="h-10 w-10 rounded-full"
                                                    src={`/storage/${account.bank.logo}`}
                                                    alt={account.bank.name}
                                                />
                                            ) : (
                                                <div
                                                    className="h-10 w-10 rounded-full bg-gray-200 flex items-center justify-center">
                                                    <span
                                                        className="text-gray-500 text-xs">{generateInital(account?.bank?.name ?? '')}</span>
                                                </div>
                                            )}
                                            <div className="flex-1 text-left">
                                                <div className="font-medium text-background">{account?.bank?.name}</div>
                                                <div className="text-sm text-muted/50">{account.account_number}</div>
                                                <div className="text-sm text-muted/50">{account.account_name}</div>
                                            </div>
                                            <Copy className="w-6 h-6 text-primary-foreground"
                                                  onClick={() => copyToClipboard(account.account_number)} />
                                        </button>
                                    ))}
                                </div>
                            </div>

                            {/* Receipt Upload */}
                            <div>
                                <h4 className="font-medium mb-3 text-background">Muat Naik Resit/Bukti Pemindahan</h4>
                                <div
                                    className={cn(
                                        'border-2 border-dashed rounded-lg h-42 relative',
                                        evidence && 'border-primary-foreground',
                                        !evidence && 'border-white hover:bg-background/30'
                                    )}
                                >
                                    <input
                                        type="file"
                                        accept="image/*"
                                        onChange={(e) => {
                                            const file = e.target.files?.[0];
                                            if (file) setEvidence(file);
                                        }}
                                        className="hidden"
                                        id="receipt-upload"
                                    />
                                    <label htmlFor="receipt-upload"
                                           className="flex flex-row h-full justify-center items-center cursor-pointer">
                                        {evidence ? (
                                            <>
                                                <Button
                                                    className="absolute top-2 right-2 z-10 rounded-full h-6 w-6 bg-red-700 hover:bg-red-800 text-white"
                                                    onClick={(e) => {
                                                        e.stopPropagation();
                                                        e.preventDefault();
                                                        setEvidence(null);
                                                    }}
                                                >
                                                    <X />
                                                </Button>
                                                <img
                                                    src={URL.createObjectURL(evidence)}
                                                    alt="Bukti bayar"
                                                    className="w-full h-full inset-0 absolute object-center object-cover rounded-lg"
                                                />
                                                <div
                                                    className="w-12 h-12 bg-accent/10 rounded-lg flex items-center justify-center">
                                                    <CheckCircle className="w-6 h-6 text-accent" />
                                                </div>
                                            </>
                                        ) : (
                                            <div className="flex flex-col items-center space-y-2">
                                                <div
                                                    className="w-12 h-12 bg-primary-foreground/20 rounded-lg flex items-center justify-center">
                                                    <Camera className="w-6 h-6 text-muted" />
                                                </div>
                                                <div className="text-center text-background">
                                                    <div className="font-medium">Tekan untuk Muat Naik</div>
                                                    <div className="text-sm text-muted">Pilih gambar resit dari galeri
                                                    </div>
                                                </div>
                                            </div>
                                        )}
                                    </label>
                                </div>

                                {/* Evidence Error Display */}
                                {error && (
                                    <div className="mt-2 bg-red-500/20 border border-red-500/50 rounded-lg p-2">
                                        <p className="text-red-400 text-xs text-center font-medium">
                                            {error}
                                        </p>
                                    </div>
                                )}
                            </div>

                            {/* Action Buttons */}
                            <div className="flex gap-3">
                                <Button
                                    variant="outline"
                                    onClick={() => {
                                        setError(null); // Clear any previous errors
                                        setStep('amount');
                                    }}
                                    className="flex-1 font-bold border border-primary-foreground bg-primary hover:bg-primary-foreground text-primary-foreground hover:text-primary"
                                >
                                    Kembali
                                </Button>
                                <Button
                                    onClick={() => {
                                        setError(null); // Clear any previous errors
                                        setStep('pin');
                                    }}
                                    disabled={!selectedBank || !evidence || isLoading}
                                    className="flex-1 bg-primary-foreground hover:bg-primary-foreground/80 text-primary font-bold"
                                >
                                    {isLoading ? <Loader2 className="w-4 h-4 animate-spin" /> : 'Hantar Bukti'}
                                </Button>
                            </div>
                        </>
                    )}

                    {/* Step 3: PIN Verification */}
                    {step === 'pin' && (
                        <>
                            <div className="text-center p-4 bg-primary-foreground/20 rounded-xl mb-6">
                                <div className="text-sm text-muted mb-1">Ringkasan Permohonan</div>
                                <div className="text-2xl font-bold text-white">
                                    RM {amount ? parseFloat(amount.replace(/\./g, '')).toLocaleString('ms-MY').replace(/,/g, '.') : '0'}
                                </div>
                                <div className="text-sm text-muted mt-2">
                                    {selectedBank?.bank?.name} - {selectedBank?.account_number}
                                </div>
                            </div>

                            <div className="text-center mb-6">
                                <p className="text-muted mb-4">
                                    Sila sahkan PIN transaksi anda untuk meneruskan permohonan tambah baki
                                </p>
                            </div>

                            <PinVerification
                                isOpen={step === 'pin'}
                                onClose={() => setStep('bank')}
                                onVerified={() => {
                                    // After PIN verification, automatically submit the transaction
                                    handleSubmit();
                                }}
                                title="Sahkan PIN untuk Tambah Baki"
                                description="Masukkan 6-digit PIN transaksi untuk mengesahkan permohonan tambah baki"
                            />
                        </>
                    )}

                    {/* Step 4: Loading */}
                    {step === 'loading' && (
                        <>
                            <div className="text-center">
                                <div
                                    className="w-24 h-24 bg-primary-foreground/20 rounded-full flex items-center justify-center mx-auto mb-6">
                                    <Loader2 className="w-12 h-12 text-primary-foreground animate-spin" />
                                </div>
                                <h3 className="text-xl font-bold mb-2 text-primary-foreground">Memproses Permohonan</h3>
                                <p className="text-muted mb-6">Sila tunggu sebentar sementara kami memproses permohonan
                                    anda...</p>
                            </div>

                            {/* Transaction Summary */}
                            <div className="space-y-4 bg-foreground/20 rounded-lg p-4">
                                <div className="flex justify-between">
                                    <span className="text-muted">Jumlah</span>
                                    <span
                                        className="font-bold text-primary-foreground">RM {parseFloat(amount).toFixed(2)}</span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-muted">Bank</span>
                                    <span
                                        className="font-medium text-primary-foreground">{selectedBank?.bank?.name}</span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-muted">Status</span>
                                    <span className="text-yellow-500 font-medium">Sedang Diproses...</span>
                                </div>
                            </div>
                        </>
                    )}

                    {/* Step 5: Success */}
                    {step === 'success' && (
                        <>
                            <div className="text-center">
                                <div
                                    className="w-24 h-24 bg-green-500/10 rounded-full flex items-center justify-center mx-auto mb-4">
                                    <CheckCircle className="w-12 h-12 text-green-500" />
                                </div>
                                <h3 className="text-xl font-bold mb-2 text-primary-foreground">Berjaya Dihantar!</h3>
                                <p className="text-muted mb-6">
                                    Permohonan tambah baki anda telah dihantar. Kami akan memproses segera.
                                </p>
                            </div>

                            {/* Transaction Summary */}
                            <div className="space-y-4 bg-foreground/20 rounded-lg p-4">
                                <div className="flex justify-between">
                                    <span className="text-muted">Jumlah</span>
                                    <span
                                        className="font-bold text-primary-foreground">RM {parseFloat(amount).toFixed(2)}</span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-muted">Bank</span>
                                    <span
                                        className="font-medium text-primary-foreground">{selectedBank?.bank?.name}</span>
                                </div>
                                <div className="flex justify-between">
                                    <span className="text-muted">Status</span>
                                    <span className="text-yellow-500 font-medium">Dalam Pemprosesan</span>
                                </div>
                            </div>

                            <Button
                                onClick={() => {
                                    setAmount('0');
                                    setEvidence(null);
                                    setSelectedBank(null);
                                    setStep('amount');
                                    setShowModal(false);
                                }}
                                className="cursor-pointer w-full bg-primary-foreground hover:bg-primary text-primary hover:text-primary-foreground border border-primary-foreground hover:border-primary-foreground font-bold"
                            >
                                Selesai
                            </Button>
                        </>
                    )}
                </div>
            </SheetContent>
        </Sheet>
    );
};

export default Topup;
