import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { InputOTP, InputOTPGroup, InputOTPSlot } from '@/components/ui/input-otp';
import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from '@/components/ui/sheet';
import { Shield, Loader2 } from 'lucide-react';
import { useApiClient } from '@/lib/api';
import { toast } from 'react-toastify';
import { usePage } from '@inertiajs/react';

interface PinVerificationProps {
    isOpen: boolean;
    onClose: () => void;
    onVerified: () => void;
    title?: string;
    description?: string;
}

const PinVerification = ({
    isOpen,
    onClose,
    onVerified,
    title = "Sahkan PIN Transaksi",
    description = "Masukkan 6-digit PIN transaksi untuk mengesahkan operasi"
}: PinVerificationProps) => {
    const [pin, setPin] = useState('');
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const apiClient = useApiClient();
    const {csrf_token} = usePage().props;

    const handleSubmit = async () => {
        if (pin.length !== 6) {
            setError('PIN mesti 6 digit');
            return;
        }

        setError(null);
        setIsLoading(true);

        try {
            const response = await apiClient.post('/api/pin/verify', {
                body: { _token: csrf_token, pin }
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    toast.success('PIN disahkan');
                    setPin(''); // Clear PIN for security
                    onVerified();
                    // Don't call handleClose() here as it goes back to previous step
                    // handleClose() is only for cancel/close actions
                } else {
                    setError(data.message || 'PIN tidak betul');
                }
            } else {
                const errorData = await response.json();
                if (errorData.errors) {
                    const firstError = Object.values(errorData.errors)[0];
                    setError(Array.isArray(firstError) ? firstError[0] : firstError);
                } else {
                    setError(errorData.message || 'PIN tidak betul');
                }
            }
        } catch (error) {
            console.error('Error verifying PIN:', error);
            setError('Ralat sistem. Sila cuba lagi.');
        } finally {
            setIsLoading(false);
        }
    };

    const handleClose = () => {
        setPin('');
        setError(null);
        onClose();
    };

    return (
        <Sheet open={isOpen} onOpenChange={handleClose}>
            <SheetContent side="bottom" className="max-w-sm mx-auto rounded-t-2xl overflow-y-auto transition-all">
                <SheetHeader>
                    <SheetTitle className="flex items-center gap-2 text-white">
                        <Shield className="w-5 h-5" />
                        {title}
                    </SheetTitle>
                    <SheetDescription className="text-white/80">
                        {description}
                    </SheetDescription>
                </SheetHeader>

                <div className="space-y-4 px-4 pb-4">
                    {/* Error Display */}
                    {error && (
                        <div className="text-center text-red-400 text-sm bg-red-400/10 p-3 rounded-lg">
                            {error}
                        </div>
                    )}

                    {/* PIN Input */}
                    <div className="flex justify-center">
                        <InputOTP
                            maxLength={6}
                            value={pin}
                            onChange={setPin}
                            disabled={isLoading}
                        >
                            <InputOTPGroup>
                                <InputOTPSlot index={0} />
                                <InputOTPSlot index={1} />
                                <InputOTPSlot index={2} />
                                <InputOTPSlot index={3} />
                                <InputOTPSlot index={4} />
                                <InputOTPSlot index={5} />
                            </InputOTPGroup>
                        </InputOTP>
                    </div>

                    {/* Action Buttons */}
                    <div className="flex flex-row space-x-2">
                        <Button
                            variant="outline"
                            onClick={handleClose}
                            className="w-full"
                            disabled={isLoading}
                        >
                            Batal
                        </Button>

                        <Button
                            onClick={handleSubmit}
                            disabled={pin.length !== 6 || isLoading}
                            className="w-full cursor-pointer"
                            variant="primary"
                        >
                            {isLoading ? (
                                <Loader2 className="w-4 h-4 animate-spin" />
                            ) : (
                                'Sahkan'
                            )}
                        </Button>
                    </div>
                </div>
            </SheetContent>
        </Sheet>
    );
};

export default PinVerification;
