import { useState, useEffect } from 'react';
import { Bell } from 'lucide-react';
import { useApiClient } from '@/lib/api';

interface SystemMessage {
    id: number;
    title: string;
    message: string;
    is_active: boolean;
}

export default function SystemMessage() {
    const [messages, setMessages] = useState<SystemMessage[]>([]);
    const [currentIndex, setCurrentIndex] = useState(0);
    const apiClient = useApiClient();

    useEffect(() => {
        const fetchMessages = async () => {
            try {
                const response = await apiClient.get('/api/system-messages');
                if (response.ok) {
                    const data = await response.json();
                    const activeMessages = data.filter((msg: SystemMessage) => msg.is_active);
                    setMessages(activeMessages);
                }
            } catch (error) {
                console.error('Failed to fetch system messages:', error);
            }
        };

        fetchMessages();
    }, [apiClient]);

    useEffect(() => {
        if (messages.length <= 1) return;

        const interval = setInterval(() => {
            setCurrentIndex((prevIndex) =>
                prevIndex === messages.length - 1 ? 0 : prevIndex + 1
            );
        }, 5000); // Change message every 5 seconds

        return () => clearInterval(interval);
    }, [messages.length]);

    if (messages.length === 0) return null

    const currentMessage = messages[currentIndex];

    return (
        <div className="bg-white w-28 text-center rounded-md p-1 relative pt-3">
            <div className="absolute -top-4 left-1/2 -translate-x-1/2 p-1 rounded-full bg-white border border-foreground">
                <Bell className="text-foreground w-4 h-4"  />
            </div>

            {/* Message Content */}
            <div className="min-h-[3rem] flex flex-col justify-center">
                <p className="text-[.6rem] text-foreground font-bold">
                    {currentMessage.title}
                </p>
                <p className="text-[.5rem] text-foreground leading-tight">
                    {currentMessage.message}
                </p>
            </div>
        </div>
    );
};
