import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from '@/components/ui/sheet';
import { Plus, Send, User } from 'lucide-react';
import { Ticket } from '@/types';
import { useApiClient } from '@/lib/api';
import { cn } from '@/lib/utils';
import { ArrowLeft, Check, MessageCircle, Paperclip, X as XIcon } from 'lucide-react';
import React, { useRef, useMemo, isValidElement, cloneElement } from 'react';
import dayjs from 'dayjs';
import { Image as ImageIcon } from 'lucide-react';
import { toast } from 'react-toastify';
import { router, usePage } from '@inertiajs/react';

type Attachment = {
    id: string;
    path: string;
    name: string;
    type: 'image';
};

type MessageProps = {
    id: number;
    content: string;
    sender: 'user' | 'admin';
    timestamp: string;
    senderName: string;
    attachments?: Attachment[];
};

type TicketStatus = 'open' | 'in_progress' | 'resolved' | 'closed';

type TicketProps = {
    id: number;
    title: string;
    description: string;
    status: TicketStatus;
    priority: 'low' | 'medium' | 'high';
    createdAt: string;
    updatedAt: string;
    attachments?: Attachment[];
    messages: MessageProps[];
};

type TriggerRender = (opts: { open: boolean; setOpen: (v: boolean) => void }) => React.ReactNode;

type TicketSystemProps = {
    /** Optional trigger button: pass an element or a render function */
    trigger?: React.ReactElement | TriggerRender;
    /** Callback when ticket system is opened */
    onTicketOpen?: () => void;
};

const getStatusColor = (status: TicketStatus) => {
    switch (status) {
        case 'open':
            return 'text-blue-500 bg-blue-500/20';
        case 'in_progress':
            return 'text-yellow-500 bg-yellow-500/10';
        case 'resolved':
            return 'text-green-500 bg-green-500/10';
        case 'closed':
            return 'text-white bg-gray-400';
        default:
            return 'text-gray-500 bg-gray-500/10';
    }
};

const getStatusText = (status: TicketStatus) => {
    switch (status) {
        case 'open':
            return 'Terbuka';
        case 'in_progress':
            return 'Dalam Proses';
        case 'resolved':
            return 'Selesai';
        case 'closed':
            return 'Ditutup';
        default:
            return 'Tidak Diketahui';
    }
};

const getPriorityColor = (priority: string) => {
    switch (priority) {
        case 'high':
            return 'text-red-500';
        case 'medium':
            return 'text-yellow-500';
        case 'low':
            return 'text-green-500';
        default:
            return 'text-gray-500';
    }
};

function composeHandlers<E extends React.SyntheticEvent>(
    ...handlers: Array<((e: E) => void) | undefined>
) {
    return (e: E) => {
        for (const h of handlers) h?.(e);
    };
}

const TicketSystem: React.FC<TicketSystemProps> = ({ trigger, onTicketOpen }) => {
    const {csrf_token} = usePage().props;
    const [showTicketList, setShowTicketList] = useState(false);
    const [selectedTicket, setSelectedTicket] = useState<TicketProps | null>(null);
    const [showNewTicket, setShowNewTicket] = useState(false);
    const [newMessage, setNewMessage] = useState('');
    const [newTicketTitle, setNewTicketTitle] = useState('');
    const [newTicketDescription, setNewTicketDescription] = useState('');
    const [newTicketPriority, setNewTicketPriority] = useState<'low' | 'medium' | 'high'>('medium');

    const [messageFiles, setMessageFiles] = useState<File[]>([]);
    const [messagePreviews, setMessagePreviews] = useState<string[]>([]);
    const [ticketFiles, setTicketFiles] = useState<File[]>([]);
    const [ticketPreviews, setTicketPreviews] = useState<string[]>([]);
    const [userTickets, setUserTickets] = useState<Ticket[]>([]);
    const [isLoadingTickets, setIsLoadingTickets] = useState(false);
    const [isSendingMessage, setIsSendingMessage] = useState(false);
    const [isCreatingTicket, setIsCreatingTicket] = useState(false);
    const [lastSentMessageId, setLastSentMessageId] = useState<number | null>(null);
    const messageImageInputRef = useRef<HTMLInputElement>(null);
    const ticketImageInputRef = useRef<HTMLInputElement>(null);
    const messagesContainerRef = useRef<HTMLDivElement>(null);
    const apiClient = useApiClient();

    // Previews for message images
    useEffect(() => {
        const urls = messageFiles.map((f) => URL.createObjectURL(f));
        setMessagePreviews(urls);
        return () => {
            urls.forEach((u) => URL.revokeObjectURL(u));
        };
    }, [messageFiles]);

    // Previews for new ticket images
    useEffect(() => {
        const urls = ticketFiles.map((f) => URL.createObjectURL(f));
        setTicketPreviews(urls);
        return () => {
            urls.forEach((u) => URL.revokeObjectURL(u));
        };
    }, [ticketFiles]);

    // Fetch tickets when sheet opens
    useEffect(() => {
        if (showTicketList && !selectedTicket && !showNewTicket) {
            fetchUserTickets();
            // Mark all tickets as read when ticket system opens
            markAllTicketsAsRead();
            // Call onTicketOpen callback when ticket system opens
            onTicketOpen?.();
        }
    }, [showTicketList, selectedTicket, showNewTicket, onTicketOpen]);

    // Auto-refresh messages every 30 seconds when ticket is selected
    useEffect(() => {
        if (!selectedTicket) return;

        const interval = setInterval(() => {
            refreshSelectedTicket();
        }, 30000); // 30 seconds

        return () => clearInterval(interval);
    }, [selectedTicket]);

    // Auto-scroll to bottom when messages change
    useEffect(() => {
        if (selectedTicket) {
            // Small delay to ensure DOM is updated
            setTimeout(() => {
                scrollToBottom();
            }, 100);
        }
    }, [selectedTicket?.messages]);

    // Auto-scroll when ticket is first selected
    useEffect(() => {
        if (selectedTicket) {
            // Small delay to ensure DOM is updated
            setTimeout(() => {
                scrollToBottom();
            }, 200);
        }
    }, [selectedTicket]);

    const fetchUserTickets = async () => {
        setIsLoadingTickets(true);
        try {
            const response = await fetch(route('tickets.index'), {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json'
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const data = await response.json();
            setUserTickets(data.data || []);
        } catch (error) {
            console.error('Failed to fetch tickets:', error);
        } finally {
            setIsLoadingTickets(false);
        }
    };

    const markAllTicketsAsRead = async () => {
        try {
            await fetch(route('tickets.mark-all-read'), {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    '_token': csrf_token
                })
            });
        } catch (error) {
            console.error('Failed to mark tickets as read:', error);
        }
    };

    const refreshSelectedTicket = async () => {
        if (!selectedTicket) return;

        try {
            const response = await apiClient.get(route('tickets.show', selectedTicket.id));

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const updatedTicket = await response.json();

            // Update the selected ticket with fresh data
            setSelectedTicket({
                id: updatedTicket.id,
                title: updatedTicket.title,
                description: updatedTicket.description,
                status: updatedTicket.status,
                priority: updatedTicket.priority,
                createdAt: updatedTicket.created_at,
                updatedAt: updatedTicket.updated_at,
                messages: updatedTicket.replies?.map((reply: any) => ({
                    id: reply.id,
                    content: reply.content,
                    sender: reply.user?.role === 'admin' ? 'admin' : 'user',
                    timestamp: reply.created_at,
                    senderName: reply.user?.name || 'Unknown',
                    attachments: reply.attachments?.map((att: any) => ({
                        id: att.name,
                        url: att.path,
                        name: att.name,
                        type: att.type as 'image'
                    }))
                })) || []
            });

            // Also update the ticket in the userTickets list
            setUserTickets(prev => prev.map(ticket =>
                ticket.id === updatedTicket.id ? updatedTicket : ticket
            ));
        } catch (error) {
            console.error('Failed to refresh ticket:', error);
        }
    };

    // File pick handlers
    const onPickMessageFiles: React.ChangeEventHandler<HTMLInputElement> = (e) => {
        const files = e.currentTarget.files;
        if (!files || files.length === 0) return;
        const picked: File[] = [];
        for (let i = 0; i < files.length; i++) {
            const f = files.item(i);
            if (f) picked.push(f);
        }
        setMessageFiles((prev) => [...prev, ...picked]);
        e.currentTarget.value = '';
    };

    const onPickTicketFiles: React.ChangeEventHandler<HTMLInputElement> = (e) => {
        const files = e.currentTarget.files;
        if (!files || files.length === 0) return;
        const picked: File[] = [];
        for (let i = 0; i < files.length; i++) {
            const f = files.item(i);
            if (f) picked.push(f);
        }
        setTicketFiles((prev) => [...prev, ...picked]);
        e.currentTarget.value = '';
    };

    const removeMessageFile = (idx: number) => setMessageFiles((prev) => prev.filter((_, i) => i !== idx));
    const removeTicketFile = (idx: number) => setTicketFiles((prev) => prev.filter((_, i) => i !== idx));

    const scrollToBottom = () => {
        if (messagesContainerRef.current) {
            messagesContainerRef.current.scrollTop = messagesContainerRef.current.scrollHeight;
        }
    };

    const handleCreateTicket = async () => {
        if (!newTicketTitle.trim() || !newTicketDescription.trim()) return;

        setIsCreatingTicket(true);

        const form = new FormData();
        if (csrf_token) form.append('_token', csrf_token.toString());
        form.append('title', newTicketTitle);
        form.append('description', newTicketDescription);
        form.append('priority', newTicketPriority);
        ticketFiles.forEach((f) => form.append('attachments[]', f));

        try {
            const response = await apiClient.post(route('tickets.store'), {
                body: form
            });

            if (response.ok) {
                setNewTicketTitle('');
                setNewTicketDescription('');
                setNewTicketPriority('medium');
                setTicketFiles([]);
                setShowNewTicket(false);
                setShowTicketList(true);
                // Refresh the ticket list
                fetchUserTickets();
            } else {
                console.error('Failed to create ticket');
            }
        } catch (error) {
            console.error('Error creating ticket:', error);
        } finally {
            setIsCreatingTicket(false);
        }
    };

    const handleSendMessage = async () => {
        if (!newMessage.trim() && messageFiles.length === 0) return;

        setIsSendingMessage(true);

        // Create optimistic message
        const optimisticMessage: MessageProps = {
            id: Date.now(), // Temporary ID
            content: newMessage,
            sender: 'user' as const,
            timestamp: new Date().toISOString(),
            senderName: 'You',
            attachments: messageFiles.map(file => ({
                id: file.name,
                path: URL.createObjectURL(file),
                name: file.name,
                type: 'image' as const
            }))
        };

        // Add optimistic message immediately
        if (selectedTicket) {
            setSelectedTicket(prev => prev ? {
                ...prev,
                messages: [...prev.messages, optimisticMessage]
            } : null);

            // Track this as the last sent message
            setLastSentMessageId(optimisticMessage.id);

            // Scroll to bottom after optimistic update
            setTimeout(() => {
                scrollToBottom();
            }, 100);
        }

        const formData = new FormData();
        if (csrf_token) formData.append('_token', csrf_token.toString());
        formData.append('content', newMessage);
        messageFiles.forEach((f) => formData.append('attachments[]', f));

        // Use Inertia.js router for CSRF-protected requests
        router.post(route('tickets.reply', selectedTicket!.id), formData, {
            onSuccess: () => {
                setNewMessage('');
                setMessageFiles([]);
                setLastSentMessageId(null);
                refreshSelectedTicket();
                toast.success('Reply sent successfully');
                setIsSendingMessage(false);
            },
            onError: (errors) => {
                console.error('Reply failed:', errors);

                // Remove the optimistic message on failure
                if (selectedTicket) {
                    setSelectedTicket(prev => prev ? {
                        ...prev,
                        messages: prev.messages.filter(msg => msg.id !== lastSentMessageId)
                    } : null);
                }
                setLastSentMessageId(null);

                // Show error message
                const errorMessage = errors.content || errors.attachments || 'Failed to send reply';
                toast.error(errorMessage);
                setIsSendingMessage(false);
            },
            onFinish: () => {
                // This runs after success or error
                setIsSendingMessage(false);
            }
        });
    };

    const resetStates = () => {
        setShowTicketList(false);
        setSelectedTicket(null);
        setShowNewTicket(false);
        setNewMessage('');
    };

    // ===== Customizable trigger =====
    const defaultTrigger = (
        <button
            type="button"
            className="cursor-pointer w-full flex items-center gap-4 px-4 py-2 bg-foreground/20 hover:bg-foreground/30 rounded-lg transition-colors text-left text-primary-foreground"
            onClick={() => setShowTicketList(true)}
            aria-expanded={showTicketList}
            aria-controls="ticket-list-panel"
        >
            <MessageCircle className="w-4 h-4 text-white" />
            <div className="font-medium">Sistem Tiket</div>
        </button>
    );

    const triggerNode = useMemo(() => {
        if (!trigger) return defaultTrigger;

        if (typeof trigger === 'function') {
            return (trigger as TriggerRender)({
                open: showTicketList,
                setOpen: setShowTicketList
            });
        }

        if (isValidElement(trigger)) {
            const el = trigger as React.ReactElement<any>;
            if (el.props?.disabled) return el;
            return cloneElement(el, {
                ...el.props,
                onClick: composeHandlers(el.props?.onClick, () => setShowTicketList(true)),
                'aria-expanded': showTicketList,
                'aria-controls': 'ticket-list-panel'
            });
        }

        return defaultTrigger;
    }, [trigger, showTicketList]);

    return (
        <>
            {triggerNode}

            {/* Ticket List Sheet */}
            <Sheet open={showTicketList && !selectedTicket && !showNewTicket}
                   onOpenChange={(open) => !open && resetStates()}>
                <SheetContent
                    side="bottom"
                    className="max-w-sm mx-auto rounded-t-2xl transition-all h-[85vh] flex flex-col"
                    id="ticket-list-panel"
                >
                    <SheetHeader className="sticky top-0">
                        <SheetTitle className="flex items-center justify-between text-white">
                            <div className="flex items-center gap-2">
                                <MessageCircle className="w-5 h-5" />
                                Sistem Tiket
                            </div>
                        </SheetTitle>
                        <SheetDescription>Senarai tiket sokongan anda</SheetDescription>
                    </SheetHeader>

                    <div className="px-4 flex flex-row justify-end">
                        <Button size="sm" variant="primary" onClick={() => setShowNewTicket(true)}>
                            <Plus className="w-4 h-4" />
                            Tiket Baru
                        </Button>
                    </div>

                    <div className="flex-1 overflow-y-auto px-4 pb-4">
                        <div className="space-y-3">
                            {isLoadingTickets ? (
                                <div className="text-center py-8 text-muted-foreground">
                                    <div
                                        className="animate-spin rounded-full h-8 w-8 border-b-2 border-white mx-auto mb-4"></div>
                                    <p>Memuatkan tiket...</p>
                                </div>
                            ) : userTickets.length === 0 ? (
                                <div className="text-center py-8 text-muted-foreground">
                                    <MessageCircle className="w-12 h-12 mx-auto mb-4 opacity-50" />
                                    <p>Tiada tiket lagi</p>
                                    <p className="text-sm">Buat tiket pertama anda untuk mendapatkan bantuan</p>
                                </div>
                            ) : (
                                userTickets.map((ticket) => {
                                    // Check if ticket has unread messages from admin
                                    const hasUnreadMessages = !ticket.is_read;

                                    return (
                                        <button
                                            key={ticket.id}
                                            onClick={async () => {
                                                // Mark ticket as read when opened
                                                if (!ticket.is_read) {
                                                    try {
                                                        await apiClient.get(route('tickets.show', ticket.id));

                                                        // Update local state to mark as read
                                                        setUserTickets(prev => prev.map(t =>
                                                            t.id === ticket.id ? { ...t, is_read: true } : t
                                                        ));
                                                    } catch (error) {
                                                        console.error('Failed to mark ticket as read:', error);
                                                    }
                                                }

                                                setSelectedTicket({
                                                    id: ticket.id,
                                                    title: ticket.title,
                                                    description: ticket.description,
                                                    status: ticket.status,
                                                    priority: ticket.priority,
                                                    createdAt: ticket.created_at,
                                                    updatedAt: ticket.updated_at,
                                                    attachments: ticket.attachments?.map(att => ({
                                                        id: att.name,
                                                        path: att.path,
                                                        name: att.name,
                                                        type: att.type as 'image'
                                                    })) || [],
                                                    messages: ticket.replies?.map(reply => ({
                                                        id: reply.id,
                                                        content: reply.content,
                                                        sender: reply.user?.role === 'admin' ? 'admin' : 'user',
                                                        timestamp: reply.created_at,
                                                        senderName: reply.user?.name || 'Unknown',
                                                        attachments: reply.attachments?.map(att => ({
                                                            id: att.name,
                                                            path: att.path,
                                                            name: att.name,
                                                            type: att.type as 'image'
                                                        }))
                                                    })) || []
                                                });
                                            }}
                                            className="w-full text-left p-4 bg-muted/20 hover:bg-muted/30 rounded-lg transition-colors relative"
                                        >
                                            {/* Unread indicator dot */}
                                            {hasUnreadMessages && (
                                                <div
                                                    className="absolute top-3 right-3 w-3 h-3 bg-red-500 rounded-full animate-pulse"></div>
                                            )}

                                            <div className="flex items-start justify-between mb-2">
                                                <div className="flex items-center gap-2 flex-1">
                                                    <h3 className="font-medium text-white line-clamp-1">{ticket.title}</h3>
                                                    {hasUnreadMessages && (
                                                        <span className="text-xs text-red-400 font-medium">(Baru)</span>
                                                    )}
                                                </div>
                                                <span
                                                    className={cn('text-xs px-2 py-1 rounded-full', getStatusColor(ticket.status))}>
                                                    {getStatusText(ticket.status)}
                                                </span>
                                            </div>
                                            <div
                                                className="text-sm text-muted line-clamp-2 mb-2">{ticket.description}</div>
                                            <div className="flex items-center justify-between text-xs">
                                                <span
                                                    className="text-muted">{dayjs(ticket.updated_at).format('DD MMM, HH:mm')}</span>
                                                <span
                                                    className={cn('font-medium capitalize', getPriorityColor(ticket.priority))}>
                                                    {ticket.priority === 'low' && 'Rendah'}
                                                    {ticket.priority === 'medium' && 'Medium'}
                                                    {ticket.priority === 'high' && 'Tinggi'}
                                                </span>
                                            </div>
                                        </button>
                                    );
                                })
                            )}
                        </div>
                    </div>
                </SheetContent>
            </Sheet>

            {/* New Ticket Sheet */}
            <Sheet open={showNewTicket} onOpenChange={(open) => !open && setShowNewTicket(false)}>
                <SheetContent side="bottom" className="max-w-sm mx-auto rounded-t-2xl overflow-y-auto transition-all">
                    <SheetHeader>
                        <SheetTitle className="flex items-center gap-2 text-white">
                            <button onClick={() => setShowNewTicket(false)}>
                                <ArrowLeft className="w-5 h-5" />
                            </button>
                            Tiket Baru
                        </SheetTitle>
                        <SheetDescription>Buat tiket sokongan baru</SheetDescription>
                    </SheetHeader>

                    <div className="space-y-4 px-4 pb-4">
                        <div>
                            <label className="text-sm font-medium text-white mb-2 block">Tajuk Tiket</label>
                            <Input
                                value={newTicketTitle}
                                onChange={(e) => setNewTicketTitle(e.target.value)}
                                placeholder="Masukkan tajuk tiket"
                                className="border-muted"
                            />
                        </div>

                        <div>
                            <label className="text-sm font-medium text-white mb-2 block">Keutamaan</label>
                            <div className="flex gap-2">
                                {(['low', 'medium', 'high'] as const).map((priority) => (
                                    <button
                                        key={priority}
                                        onClick={() => setNewTicketPriority(priority)}
                                        className={cn(
                                            'px-3 py-2 rounded-lg text-sm font-medium transition-colors',
                                            newTicketPriority === priority
                                                ? 'bg-primary-foreground text-primary'
                                                : 'bg-muted/20 text-muted-background hover:bg-muted/30'
                                        )}
                                    >
                                        {priority === 'low' && 'Rendah'}
                                        {priority === 'medium' && 'Medium'}
                                        {priority === 'high' && 'Tinggi'}
                                    </button>
                                ))}
                            </div>
                        </div>

                        <div>
                            <label className="text-sm font-medium text-white mb-2 block">Penerangan</label>
                            <textarea
                                value={newTicketDescription}
                                onChange={(e) => setNewTicketDescription(e.target.value)}
                                placeholder="Terangkan masalah atau pertanyaan anda"
                                rows={4}
                                className="w-full p-3 bg-background border border-muted rounded-lg text-foreground placeholder-muted-foreground resize-none"
                            />

                            <div className="space-y-2">
                                <div className="flex items-center justify-between">
                                    <span className="text-sm font-medium text-white">Lampiran Gambar</span>
                                    <div className="flex items-center gap-2">
                                        <Button
                                            type="button"
                                            size="sm"
                                            variant="outline"
                                            className="gap-2"
                                            onClick={() => ticketImageInputRef.current?.click()}
                                        >
                                            <Paperclip className="w-4 h-4" />
                                            Tambah Gambar
                                        </Button>
                                        <input
                                            ref={ticketImageInputRef}
                                            id="ticket-images"
                                            type="file"
                                            accept="image/*"
                                            multiple
                                            className="hidden"
                                            onChange={onPickTicketFiles}
                                        />
                                    </div>
                                </div>

                                {ticketPreviews.length > 0 && (
                                    <div className="grid grid-cols-4 gap-2">
                                        {ticketPreviews.map((src, idx) => (
                                            <div key={src} className="relative group">
                                                <img src={src} alt="" className="w-full h-20 object-cover rounded-md" />
                                                <button
                                                    type="button"
                                                    onClick={() => removeTicketFile(idx)}
                                                    className="absolute -top-2 -right-2 rounded-full bg-black/60 p-1 opacity-0 group-hover:opacity-100 transition"
                                                    aria-label="Remove image"
                                                >
                                                    <XIcon className="w-3 h-3 text-white" />
                                                </button>
                                            </div>
                                        ))}
                                    </div>
                                )}
                            </div>
                        </div>

                        <div className="flex gap-3 pt-4">
                            <Button variant="outline" onClick={() => setShowNewTicket(false)} className="flex-1">
                                Batal
                            </Button>
                            <Button
                                onClick={handleCreateTicket}
                                disabled={!newTicketTitle.trim() || !newTicketDescription.trim() || isCreatingTicket}
                                className="flex-1"
                                variant="primary"
                            >
                                {isCreatingTicket ? (
                                    <>
                                        <div
                                            className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                        Membuat...
                                    </>
                                ) : (
                                    'Buat Tiket'
                                )}
                            </Button>
                        </div>
                    </div>
                </SheetContent>
            </Sheet>

            {/* Ticket Detail Sheet */}
            <Sheet open={!!selectedTicket} onOpenChange={(open) => !open && setSelectedTicket(null)}>
                <SheetContent side="bottom"
                              className="max-w-sm mx-auto rounded-t-2xl overflow-y-auto transition-all h-[90vh]">
                    {selectedTicket && (
                        <>
                            <SheetHeader className="border-b border-muted/20 pb-4">
                                <SheetTitle className="flex items-center gap-2 text-white">
                                    <button onClick={() => setSelectedTicket(null)}>
                                        <ArrowLeft className="w-5 h-5" />
                                    </button>
                                    <div className="flex-1 text-left space-y-1">
                                        <div className="line-clamp-1">{selectedTicket.title}</div>
                                        <div className="flex flex-row space-x-2 items-center">
                                            <div className="text-xs text-muted-foreground font-normal">Tiket
                                                #{selectedTicket.id}</div>
                                            <span
                                                className={cn('text-xs px-2 py-1 rounded-full', getStatusColor(selectedTicket.status))}>
                        {getStatusText(selectedTicket.status)}
                      </span>
                                        </div>
                                    </div>
                                </SheetTitle>
                            </SheetHeader>

                            <div ref={messagesContainerRef} className="flex-1 px-4 py-4 space-y-4 overflow-y-auto">
                                {/* Initial ticket message */}
                                <div className="flex gap-3 flex-row-reverse">
                                    <div
                                        className="w-8 h-8 rounded-full flex items-center justify-center flex-shrink-0 bg-primary-foreground text-primary">
                                        <User className="w-4 h-4" />
                                    </div>
                                    <div className="max-w-[80%] space-y-1 items-start">
                                        <div className="text-xs text-muted-foreground">
                                            You • {dayjs(selectedTicket.createdAt).format('DD MMM, HH:mm')}
                                        </div>
                                        <div className="p-3 rounded-lg text-sm bg-primary-foreground text-primary rounded-bl-sm">
                                            <div className="font-medium mb-2">{selectedTicket.title}</div>
                                            {selectedTicket.description}
                                            {selectedTicket.attachments?.length ? (
                                                <div className="mt-3 grid grid-cols-2 gap-2">
                                                    {selectedTicket.attachments.map((att) => (
                                                        <a key={att.id} href={att.path.startsWith('blob:') ? att.path : `/storage/${att.path}`} target="_blank"
                                                           rel="noreferrer">
                                                            <img
                                                                src={att.path.startsWith('blob:') ? att.path : `/storage/${att.path}`}
                                                                alt={att.name}
                                                                className="w-full h-28 object-cover rounded-md" />
                                                        </a>
                                                    ))}
                                                </div>
                                            ) : null}
                                        </div>
                                    </div>
                                </div>
                                {selectedTicket.messages.map((message) => (
                                    <div
                                        key={message.id}
                                        className={cn('flex gap-3', message.sender === 'user' ? 'flex-row-reverse' : 'flex-row')}
                                    >
                                        <div
                                            className={cn(
                                                'w-8 h-8 rounded-full flex items-center justify-center flex-shrink-0',
                                                message.sender === 'user' ? 'bg-primary-foreground text-primary' : 'bg-muted text-muted-foreground'
                                            )}
                                        >
                                            <User className="w-4 h-4" />
                                        </div>
                                        <div
                                            className={cn('max-w-[80%] space-y-1', message.sender === 'user' ? 'items-end' : 'items-start')}>
                                            <div className="text-xs text-muted-foreground">
                                                {message.senderName} • {dayjs(message.timestamp).format('DD MMM, HH:mm')}
                                            </div>
                                            <div
                                                className={cn(
                                                    'p-3 rounded-lg text-sm',
                                                    message.sender === 'user'
                                                        ? 'bg-primary-foreground text-primary rounded-br-sm'
                                                        : 'bg-muted/30 text-white rounded-bl-sm'
                                                )}
                                            >
                                                {message.content}
                                                {message.attachments?.length ? (
                                                    <div className="mt-2 grid grid-cols-2 gap-2">
                                                        {message.attachments.map((att) => (
                                                            <a key={att.id}
                                                               href={att.path.startsWith('blob:') ? att.path : `/storage/${att.path}`}
                                                               target="_blank"
                                                               rel="noreferrer">
                                                                <img
                                                                    src={att.path.startsWith('blob:') ? att.path : `/storage/${att.path}`}
                                                                    alt={att.name}
                                                                    className="w-full h-28 object-cover rounded-md" />
                                                            </a>
                                                        ))}
                                                    </div>
                                                ) : null}

                                                {/* Send indicator for user messages */}
                                                {message.sender === 'user' && (
                                                    <div className="flex items-center justify-end mt-2">
                                                        {lastSentMessageId === message.id ? (
                                                            <div className="flex items-center gap-1">
                                                                <div
                                                                    className="w-2 h-2 bg-primary/60 rounded-full animate-bounce"></div>
                                                                <div
                                                                    className="w-2 h-2 bg-primary/60 rounded-full animate-bounce"
                                                                    style={{ animationDelay: '0.1s' }}></div>
                                                                <div
                                                                    className="w-2 h-2 bg-primary/60 rounded-full animate-bounce"
                                                                    style={{ animationDelay: '0.2s' }}></div>
                                                            </div>
                                                        ) : (
                                                            <Check className="w-4 h-4 text-blue-600" />
                                                        )}
                                                    </div>
                                                )}
                                            </div>
                                        </div>
                                    </div>
                                ))}
                            </div>

                            {selectedTicket.status !== 'closed' && (
                                <div className="border-t border-muted/20 p-4 space-y-3">
                                    {messagePreviews.length > 0 && (
                                        <div className="grid grid-cols-4 gap-2">
                                            {messagePreviews.map((src, idx) => (
                                                <div key={src} className="relative group">
                                                    <img src={src} alt=""
                                                         className="w-full h-20 object-cover rounded-md" />
                                                    <button
                                                        type="button"
                                                        onClick={() => removeMessageFile(idx)}
                                                        className="absolute -top-2 -right-2 rounded-full bg-black/60 p-1 opacity-0 group-hover:opacity-100 transition"
                                                        aria-label="Remove image"
                                                    >
                                                        <XIcon className="w-3 h-3 text-white" />
                                                    </button>
                                                </div>
                                            ))}
                                        </div>
                                    )}

                                    {['open', 'in_progress'].includes(selectedTicket.status) && (
                                        <div className="flex gap-2">
                                            <Input
                                                value={newMessage}
                                                onChange={(e) => setNewMessage(e.target.value)}
                                                placeholder="Tulis mesej..."
                                                className="flex-1 border-muted"
                                                onKeyPress={(e) => {
                                                    if (e.key === 'Enter' && !e.shiftKey) {
                                                        e.preventDefault();
                                                        handleSendMessage();
                                                    }
                                                }}
                                            />

                                            <Button type="button" size="icon" variant="outline"
                                                    onClick={() => messageImageInputRef.current?.click()}>
                                                <ImageIcon className="w-4 h-4" />
                                            </Button>
                                            <input
                                                ref={messageImageInputRef}
                                                id="message-images"
                                                type="file"
                                                accept="image/*"
                                                multiple
                                                className="hidden"
                                                onChange={onPickMessageFiles}
                                            />

                                            <Button
                                                onClick={handleSendMessage}
                                                disabled={(!newMessage.trim() && messageFiles.length === 0) || isSendingMessage}
                                                size="icon"
                                                variant="primary"
                                            >
                                                {isSendingMessage ? (
                                                    <div
                                                        className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                                                ) : (
                                                    <Send className="w-4 h-4" />
                                                )}
                                            </Button>
                                        </div>
                                    )}
                                </div>
                            )}
                        </>
                    )}
                </SheetContent>
            </Sheet>
        </>
    );
};

export default TicketSystem;
