import { Button } from '@/components/ui/button';
import { InputOTP, InputOTPGroup, InputOTPSlot } from '@/components/ui/input-otp';
import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from '@/components/ui/sheet';
import { Shield, Loader2 } from 'lucide-react';
import { useState, useEffect } from 'react';
import { useApiClient } from '@/lib/api';
import { toast } from 'react-toastify';
import { usePage } from '@inertiajs/react';

interface ChangePinProps {
    user: {
        id: number;
        pin_hash?: string | null;
    };
    hasPin?: boolean;
    forceOpen?: boolean;
    onPinSet?: () => void;
}

const ChangePin = ({ user, hasPin, forceOpen = false, onPinSet }: ChangePinProps) => {
    const {csrf_token} = usePage().props;
    const [showModal, setShowModal] = useState(forceOpen);
    const [pinStep, setPinStep] = useState<'old' | 'new' | 'confirm'>('old');
    const [oldPin, setOldPin] = useState('');
    const [newPin, setNewPin] = useState('');
    const [confirmPin, setConfirmPin] = useState('');
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [isValidatingOldPin, setIsValidatingOldPin] = useState(false);
    const [oldPinValid, setOldPinValid] = useState(false);
    const apiClient = useApiClient();

    // If user has no PIN, start with 'new' step
    useEffect(() => {
        if (!hasPin) {
            setPinStep('new');
        }
    }, [hasPin]);

    // Handle force open for users without PIN
    useEffect(() => {
        if (forceOpen && !hasPin) {
            setShowModal(true);
        }
    }, [forceOpen, hasPin]);

    const handleClose = () => {
        // If user has no PIN and this is forced open, don't allow closing
        if (forceOpen && !hasPin) {
            return;
        }

        setShowModal(false);
        resetForm();
    };

    const resetForm = () => {
        setOldPin('');
        setNewPin('');
        setConfirmPin('');
        setError(null);
        setOldPinValid(false);
        setIsValidatingOldPin(false);
        setPinStep(hasPin ? 'old' : 'new');
    };

    const handleSubmit = async () => {
        setError(null);
        setIsLoading(true);

        try {
            if (!hasPin) {
                // Creating new PIN
                if (pinStep === 'new') {
                    // Move to confirm step
                    setPinStep('confirm');
                    setIsLoading(false);
                    return;
                } else if (pinStep === 'confirm') {
                    // Validate PINs match and submit
                    if (newPin !== confirmPin) {
                        setError('PIN tidak sepadan');
                        setIsLoading(false);
                        return;
                    }

                    const response = await apiClient.post('/api/pin/create', {
                        body: { _token: csrf_token, pin: newPin, confirm_pin: confirmPin }
                    });

                    if (response.ok) {
                        const data = await response.json();
                        if (data.success) {
                            toast.success(data.message);
                            setShowModal(false);
                            resetForm();
                            onPinSet?.();
                        } else {
                            setError(data.message || 'Gagal mencipta PIN');
                        }
                    } else {
                        const errorData = await response.json();
                        if (errorData.errors) {
                            const firstError = Object.values(errorData.errors)[0];
                            setError(Array.isArray(firstError) ? firstError[0] : firstError);
                        } else {
                            setError(errorData.message || 'Gagal mencipta PIN');
                        }
                    }
                }
            } else {
                // Updating existing PIN
                if (pinStep === 'new') {
                    // Move to confirm step
                    setPinStep('confirm');
                    setIsLoading(false);
                    return;
                } else if (pinStep === 'confirm') {
                    // Validate PINs match and submit
                    if (newPin !== confirmPin) {
                        setError('PIN tidak sepadan');
                        setIsLoading(false);
                        return;
                    }

                    const response = await apiClient.post('/api/pin/update', {
                        body: {
                            _token: csrf_token,
                            old_pin: oldPin,
                            new_pin: newPin,
                            confirm_pin: confirmPin
                        }
                    });

                    if (response.ok) {
                        const data = await response.json();
                        if (data.success) {
                            toast.success(data.message);
                            setShowModal(false);
                            resetForm();
                            onPinSet?.();
                        } else {
                            setError(data.message || 'Gagal mengemaskini PIN');
                        }
                    } else {
                        const errorData = await response.json();
                        if (errorData.errors) {
                            const firstError = Object.values(errorData.errors)[0];
                            setError(Array.isArray(firstError) ? firstError[0] : firstError);
                        } else {
                            setError(errorData.message || 'Gagal mengemaskini PIN');
                        }
                    }
                }
            }
        } catch (error) {
            console.error('Error handling PIN:', error);
            setError('Ralat sistem. Sila cuba lagi.');
        } finally {
            setIsLoading(false);
        }
    };

    const handleStepChange = (step: 'old' | 'new' | 'confirm') => {
        setError(null);
        setPinStep(step);
    };

    const validateOldPin = async (pin: string) => {
        if (pin.length !== 6) return;

        setIsValidatingOldPin(true);
        setError(null);

        try {
            const response = await apiClient.post('/api/pin/verify', {
                body: { _token: csrf_token, pin }
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    setOldPinValid(true);
                    setError(null);
                    // Auto-advance to next step
                    setPinStep('new');
                } else {
                    setOldPinValid(false);
                    setError(data.message || 'PIN tidak betul');
                }
            } else {
                const errorData = await response.json();
                if (errorData.errors) {
                    const firstError = Object.values(errorData.errors)[0];
                    setError(Array.isArray(firstError) ? firstError[0] : firstError);
                } else {
                    setError(errorData.message || 'PIN tidak betul');
                }
                setOldPinValid(false);
            }
        } catch (error) {
            console.error('Error validating old PIN:', error);
            setError('Ralat sistem. Sila cuba lagi.');
            setOldPinValid(false);
        } finally {
            setIsValidatingOldPin(false);
        }
    };

    const canProceed = () => {
        if (!hasPin) {
            // For new PIN creation
            if (pinStep === 'new') {
                return newPin.length === 6;
            } else if (pinStep === 'confirm') {
                return confirmPin.length === 6;
            }
        } else {
            // For existing PIN updates
            if (pinStep === 'old') {
                return oldPin.length === 6 && !isValidatingOldPin;
            } else if (pinStep === 'new') {
                return newPin.length === 6;
            } else if (pinStep === 'confirm') {
                return confirmPin.length === 6;
            }
        }
        return false;
    };

    const getStepTitle = () => {
        if (!hasPin) {
            return pinStep === 'new' ? 'Buat PIN Baru' : 'Sahkan PIN';
        }

        switch (pinStep) {
            case 'old': return 'Masukkan PIN Semasa';
            case 'new': return 'Masukkan PIN Baru';
            case 'confirm': return 'Sahkan PIN Baru';
            default: return 'Pengurusan PIN';
        }
    };

    const getStepDescription = () => {
        if (!hasPin) {
            return pinStep === 'new'
                ? 'Cipta 6-digit PIN transaksi untuk keselamatan'
                : 'Masukkan semula PIN untuk pengesahan';
        }

        switch (pinStep) {
            case 'old': return 'Masukkan PIN 6-digit semasa anda';
            case 'new': return 'Cipta PIN 6-digit baru';
            case 'confirm': return 'Masukkan semula PIN baru untuk pengesahan';
            default: return '';
        }
    };

    return (
        <>
            {!forceOpen && (
                <button
                    className="cursor-pointer w-full flex items-center gap-4 px-4 py-2 bg-foreground/20 hover:bg-foreground/30 rounded-lg transition-colors text-left text-primary-foreground"
                    onClick={() => setShowModal(true)}
                >
                    <Shield className="w-4 h-4 text-white" />
                    <div className="font-medium">
                        {hasPin ? 'Tukar PIN Transaksi' : 'Buat PIN Transaksi'}
                    </div>
                </button>
            )}

            <Sheet open={showModal} onOpenChange={handleClose}>
                <SheetContent side="bottom" className="max-w-sm mx-auto rounded-t-2xl overflow-y-auto transition-all">
                    <SheetHeader>
                        <SheetTitle className="flex items-center gap-2 text-white">
                            <Shield className="w-5 h-5" />
                            {getStepTitle()}
                        </SheetTitle>
                        <SheetDescription className="text-white/80">
                            {getStepDescription()}
                        </SheetDescription>
                    </SheetHeader>

                    <div className="space-y-4 px-4 pb-4">
                        {/* Step indicator */}
                        <div className="flex justify-center gap-2">
                            {hasPin && (
                                <div className={`w-2 h-2 rounded-full ${pinStep === 'old' ? 'bg-white' : 'bg-muted'}`} />
                            )}
                            <div className={`w-2 h-2 rounded-full ${pinStep === 'new' ? 'bg-white' : 'bg-muted'}`} />
                            <div className={`w-2 h-2 rounded-full ${pinStep === 'confirm' ? 'bg-white' : 'bg-muted'}`} />
                        </div>

                        {/* Error Display */}
                        {error && (
                            <div className="text-center text-red-400 text-sm bg-red-400/10 p-3 rounded-lg">
                                {error}
                            </div>
                        )}

                        {/* Real-time PIN mismatch validation */}
                        {pinStep === 'confirm' && newPin.length === 6 && confirmPin.length === 6 && newPin !== confirmPin && (
                            <div className="text-center text-red-400 text-sm bg-red-400/10 p-3 rounded-lg">
                                PIN tidak sepadan. Sila cuba lagi.
                            </div>
                        )}

                        {/* PIN Input */}
                        <div className="flex justify-center">
                            <InputOTP
                                maxLength={6}
                                value={
                                    pinStep === 'old' ? oldPin :
                                    pinStep === 'new' ? newPin :
                                    confirmPin
                                }
                                onChange={(value) => {
                                    if (pinStep === 'old') {
                                        setOldPin(value);
                                        // Validate old PIN when 6 digits are entered
                                        if (value.length === 6) {
                                            validateOldPin(value);
                                        }
                                    } else if (pinStep === 'new') setNewPin(value);
                                    else setConfirmPin(value);
                                }}
                                disabled={isLoading || (pinStep === 'old' && isValidatingOldPin)}
                            >
                                <InputOTPGroup>
                                    <InputOTPSlot index={0} />
                                    <InputOTPSlot index={1} />
                                    <InputOTPSlot index={2} />
                                    <InputOTPSlot index={3} />
                                    <InputOTPSlot index={4} />
                                    <InputOTPSlot index={5} />
                                </InputOTPGroup>
                            </InputOTP>
                        </div>

                        {/* Loading indicator for old PIN validation */}
                        {pinStep === 'old' && isValidatingOldPin && (
                            <div className="text-center text-blue-400 text-sm bg-blue-400/10 p-3 rounded-lg">
                                <Loader2 className="w-4 h-4 animate-spin inline mr-2" />
                                Mengesahkan PIN...
                            </div>
                        )}

                        {/* Action Buttons */}
                        <div className="flex flex-row space-x-2">
                            {pinStep !== 'old' && !forceOpen && (
                                <Button
                                    variant="outline"
                                    onClick={() => {
                                        if (pinStep === 'new') {
                                            handleStepChange('old');
                                            setNewPin('');
                                        } else if (pinStep === 'confirm') {
                                            handleStepChange('new');
                                            setConfirmPin('');
                                        }
                                    }}
                                    className="w-full"
                                    disabled={isLoading}
                                >
                                    Kembali
                                </Button>
                            )}

                            <Button
                                onClick={handleSubmit}
                                disabled={!canProceed() || isLoading}
                                className="w-full cursor-pointer"
                                variant="primary"
                            >
                                {isLoading ? (
                                    <Loader2 className="w-4 h-4 animate-spin" />
                                ) : (
                                    (() => {
                                        if (!hasPin) {
                                            // For new PIN creation
                                            if (pinStep === 'new') return 'Teruskan';
                                            if (pinStep === 'confirm') return 'Buat PIN';
                                        } else {
                                            // For existing PIN updates
                                            if (pinStep === 'old') return isValidatingOldPin ? 'Mengesahkan...' : 'Sahkan PIN';
                                            if (pinStep === 'new') return 'Teruskan';
                                            if (pinStep === 'confirm') return 'Kemaskini PIN';
                                        }
                                        return 'Teruskan';
                                    })()
                                )}
                            </Button>
                        </div>
                    </div>
                </SheetContent>
            </Sheet>
        </>
    );
};

export default ChangePin;
