import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Key } from 'lucide-react';
import { useApiClient } from '@/lib/api';
import { toast } from 'react-toastify';
import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from '@/components/ui/sheet';
import { usePage } from '@inertiajs/react';

const ChangePassword = () => {
    const [showModal, setShowModal] = useState(false);
    const [currentPassword, setCurrentPassword] = useState('');
    const [newPassword, setNewPassword] = useState('');
    const [confirmPassword, setConfirmPassword] = useState('');
    const [isLoading, setIsLoading] = useState(false);
    const [errors, setErrors] = useState<Record<string, string[]>>({});
    const apiClient = useApiClient();
    const { csrf_token } = usePage().props;

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();

        if (newPassword !== confirmPassword) {
            toast.error('New password and confirmation do not match');
            return;
        }

        setIsLoading(true);

        try {
            const formData = new FormData();
            if (csrf_token)
                formData.append('_token', csrf_token.toString());
            formData.append('current_password', currentPassword);
            formData.append('password', newPassword);
            formData.append('password_confirmation', confirmPassword);

            const response = await apiClient.post(route('password.update'), {
                body: formData
            });

            if (response.ok) {
                const successData = await response.json();
                toast.success(successData.message || 'Password updated successfully!');
                resetForm();
            } else {
                const errorData = await response.json();
                console.log('Error response:', errorData); // Debug log
                if (errorData.errors) {
                    console.log('Setting errors:', errorData.errors); // Debug log
                    setErrors(errorData.errors);
                } else {
                    toast.error(errorData.message || 'Failed to update password');
                }
            }
        } catch (error) {
            console.error('Error updating password:', error);
            toast.error('An error occurred while updating password');
        } finally {
            setIsLoading(false);
        }
    };

    const resetForm = () => {
        setCurrentPassword('');
        setNewPassword('');
        setConfirmPassword('');
        setErrors({});
        setShowModal(false);
    };

    return (
        <>
            <button
                className="cursor-pointer w-full flex items-center gap-4 px-4 py-2 bg-foreground/20 hover:bg-foreground/30 rounded-lg transition-colors text-left text-primary-foreground"
                onClick={() => setShowModal(true)}
            >
                <Key className="w-4 h-4 text-white" />
                <div className="font-medium">Tukar Kata Laluan</div>
            </button>
            <Sheet open={showModal} onOpenChange={setShowModal}>
                <SheetContent side="bottom" className="max-w-sm mx-auto rounded-t-2xl overflow-y-auto transition-all">
                    <SheetHeader>
                        <SheetTitle className="flex items-center gap-2 text-white">
                            <Key className="w-5 h-5 text-yellow-500" />
                            Tukar Kata Laluan
                        </SheetTitle>
                        <SheetDescription className="text-background">
                            Update your login password for security
                        </SheetDescription>
                    </SheetHeader>

                    <form onSubmit={handleSubmit} className="space-y-4 px-4 pb-4">
                        <div className="flex flex-col space-y-2">
                            <label className="text-background text-sm font-medium">Current Password</label>
                            <Input
                                type="password"
                                placeholder="Enter current password"
                                value={currentPassword}
                                onChange={(e) => setCurrentPassword(e.target.value)}
                                required
                                className={errors.current_password ? 'border-red-500' : ''}
                            />
                            {errors.current_password && (
                                <div className="text-red-500 text-xs">
                                    {errors.current_password[0]}
                                </div>
                            )}
                        </div>

                        <div className="flex flex-col space-y-2">
                            <label className="text-background text-sm font-medium">New Password</label>
                            <Input
                                type="password"
                                placeholder="Enter new password"
                                value={newPassword}
                                onChange={(e) => setNewPassword(e.target.value)}
                                required
                                className={errors.password ? 'border-red-500' : ''}
                            />
                            {errors.password && (
                                <div className="text-red-500 text-xs">
                                    {errors.password[0]}
                                </div>
                            )}
                        </div>

                        <div className="flex flex-col space-y-2">
                            <label className="text-background text-sm font-medium">Confirm New Password</label>
                            <Input
                                type="password"
                                placeholder="Confirm new password"
                                value={confirmPassword}
                                onChange={(e) => setConfirmPassword(e.target.value)}
                                required
                                className={errors.password_confirmation ? 'border-red-500' : ''}
                            />
                            {errors.password_confirmation && (
                                <div className="text-red-500 text-xs">
                                    {errors.password_confirmation[0]}
                                </div>
                            )}
                        </div>

                        <div className="flex gap-3 pt-4">
                            <Button
                                type="button"
                                variant="outline"
                                className="flex-1"
                                onClick={resetForm}
                                disabled={isLoading}
                            >
                                Cancel
                            </Button>
                            <Button
                                type="submit"
                                className="flex-1"
                                variant="primary"
                                disabled={isLoading}
                            >
                                {isLoading ? (
                                    <>
                                        <div
                                            className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                        Updating...
                                    </>
                                ) : (
                                    'Update Password'
                                )}
                            </Button>
                        </div>
                    </form>
                </SheetContent>
            </Sheet>
        </>
    );
};

export default ChangePassword;
