import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Landmark, Save } from 'lucide-react';
import { Bank, BankAccount } from '@/types';
import { useApiClient } from '@/lib/api';
import { usePage } from '@inertiajs/react';

interface BankAccountFormProps {
    bankAccount?: BankAccount;
    onSuccess: () => void;
    onCancel: () => void;
}

const BankAccountForm = ({ bankAccount, onSuccess, onCancel }: BankAccountFormProps) => {
    const { csrf_token } = usePage().props;
    const [isLoading, setIsLoading] = useState(false);
    const [banks, setBanks] = useState<Bank[]>([]);
    const [formData, setFormData] = useState({
        bank_id: bankAccount?.bank_id?.toString() || '',
        account_number: bankAccount?.account_number || '',
        account_name: bankAccount?.account_name || ''
    });
    const apiClient = useApiClient();

    useEffect(() => {
        fetchBanks();
    }, []);

    const fetchBanks = async () => {
        try {
            const response = await apiClient.get('/api/banks');
            if (response.ok) {
                const data = await response.json();
                setBanks(data);
            }
        } catch (error) {
            console.error('Error fetching banks:', error);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setIsLoading(true);

        try {
            const url = bankAccount ? `/bank-accounts/${bankAccount.id}` : '/bank-accounts';
            const method = bankAccount ? 'PUT' : 'POST';

            const response = await apiClient.request(url, {
                method: method as any,
                body: { _token: csrf_token, ...formData }
            });

            if (response.ok) {
                onSuccess();
            } else {
                const errorData = await response.json();
                console.error('Error:', errorData);
            }
        } catch (error) {
            console.error('Error submitting form:', error);
        } finally {
            setIsLoading(false);
        }
    };

    const handleInputChange = (field: string, value: string) => {
        setFormData(prev => ({
            ...prev,
            [field]: value
        }));
    };

    return (
        <form onSubmit={handleSubmit} className="space-y-4">
            <div className="space-y-2">
                <Label htmlFor="bank_id" className="text-white">Bank</Label>
                <Select
                    value={formData.bank_id}
                    onValueChange={(value) => handleInputChange('bank_id', value)}
                >
                    <SelectTrigger className="bg-foreground/20 border-border text-white">
                        <SelectValue placeholder="Select a bank" />
                    </SelectTrigger>
                    <SelectContent className="border-border">
                        {banks.map((bank) => (
                            <SelectItem key={bank.id} value={bank.id.toString()}>
                                <div className="flex items-center gap-2">
                                    {bank.logo ? (
                                        <img
                                            src={`/storage/${bank.logo}`}
                                            alt={bank.name}
                                            className="w-4 h-4 rounded"
                                        />
                                    ) : (
                                        <Landmark className="w-4 h-4" />
                                    )}
                                    {bank.name}
                                </div>
                            </SelectItem>
                        ))}
                    </SelectContent>
                </Select>
            </div>

            <div className="space-y-2">
                <Label htmlFor="account_number" className="text-white">Account Number</Label>
                <Input
                    id="account_number"
                    type="text"
                    value={formData.account_number}
                    onChange={(e) => handleInputChange('account_number', e.target.value)}
                    placeholder="Enter account number"
                    className="bg-foreground/20 border-border text-white placeholder:text-muted"
                    required
                />
            </div>

            <div className="space-y-2">
                <Label htmlFor="account_name" className="text-white">Account Name</Label>
                <Input
                    id="account_name"
                    type="text"
                    value={formData.account_name}
                    onChange={(e) => handleInputChange('account_name', e.target.value)}
                    placeholder="Enter account holder name"
                    className="bg-foreground/20 border-border text-white placeholder:text-muted"
                    required
                />
            </div>

            <div className="flex gap-3 pt-4">
                <Button
                    type="button"
                    variant="outline"
                    onClick={onCancel}
                    className="flex-1 bg-transparent border-border text-white hover:bg-foreground/20"
                    disabled={isLoading}
                >
                    Cancel
                </Button>
                <Button
                    type="submit"
                    className="flex-1 bg-primary-foreground hover:bg-primary-foreground/80 text-primary"
                    disabled={isLoading}
                >
                    {isLoading ? (
                        <Save className="w-4 h-4 animate-spin" />
                    ) : (
                        bankAccount ? 'Update Account' : 'Add Account'
                    )}
                </Button>
            </div>
        </form>
    );
};

export default BankAccountForm;
