import { useState, useEffect, ReactNode } from 'react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Trash2, Plus, Edit, Eye, EyeOff, Landmark, Loader2 } from 'lucide-react';
import { BankAccount } from '@/types';
import { useApiClient } from '@/lib/api';
import BankAccountForm from './bank-account-form';
import {
    Sheet,
    SheetContent,
    SheetDescription,
    SheetHeader,
    SheetTitle,
    SheetTrigger,
} from '@/components/ui/sheet';
import { usePage } from '@inertiajs/react';
import Swal from 'sweetalert2';

interface AccountBankProps {
    /** Custom element that opens the sheet. Example: <Button>Open</Button> */
    trigger?: ReactNode;
    /** Optional: change where the sheet slides from (defaults to 'bottom') */
    side?: 'bottom' | 'top' | 'left' | 'right';
    /** Optional: extend SheetContent className */
    contentClassName?: string;
    onSuccess?: () => void
}

const AccountBank = ({ trigger, side = 'bottom', contentClassName = '', onSuccess }: AccountBankProps) => {
    const { csrf_token } = usePage().props;
    const [showModal, setShowModal] = useState(false);
    const [showForm, setShowForm] = useState(false);
    const [editingAccount, setEditingAccount] = useState<BankAccount | null>(null);
    const [bankAccounts, setBankAccounts] = useState<BankAccount[]>([]);
    const [isLoading, setIsLoading] = useState(false);
    const apiClient = useApiClient();

    useEffect(() => {
        if (showModal) {
            fetchBankAccounts();
        }
    }, [showModal]);

    const fetchBankAccounts = async () => {
        setIsLoading(true);
        try {
            const response = await apiClient.get('/bank-accounts');
            if (response.ok) {
                const data = await response.json();
                setBankAccounts(data.bank_accounts || []);
            }
        } catch (error) {
            console.error('Error fetching bank accounts:', error);
        } finally {
            setIsLoading(false);
        }
    };

    const handleToggleStatus = async (account: BankAccount) => {
        try {
            const response = await apiClient.patch(`/bank-accounts/${account.id}/toggle-status`, {
                body: JSON.stringify({ _token: csrf_token }),
            });

            if (response.ok) {
                await fetchBankAccounts();
            }
        } catch (error) {
            console.error('Error toggling status:', error);
        }
    };

    const handleDelete = async (account: BankAccount) => {
        const { isConfirmed } = await Swal.fire({
            title: 'Delete bank account?',
            html: `
        <div class="text-left">
          <div>Bank: <b>${account.bank?.name ?? '-'}</b></div>
          <div>No. Rekening: <code>${account.account_number}</code></div>
        </div>
      `,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes, delete',
            cancelButtonText: 'Cancel',
            reverseButtons: true,
            focusCancel: true,
        });
        if (!isConfirmed) return;

        // Loader (non-blocking)
        Swal.fire({
            title: 'Deleting...',
            allowOutsideClick: false,
            allowEscapeKey: false,
            showConfirmButton: false,
            didOpen: () => Swal.showLoading(),
        });

        try {
            const response = await apiClient.delete(`/bank-accounts/${account.id}`, {
                headers: {
                    'X-CSRF-TOKEN': csrf_token,
                    'X-XSRF-TOKEN': csrf_token,
                },
            });

            Swal.close();

            if (response.ok) {
                await fetchBankAccounts();
                await Swal.fire({
                    icon: 'success',
                    title: 'Deleted',
                    text: 'Bank account has been removed.',
                    timer: 1400,
                    showConfirmButton: false,
                });
            } else {
                let message = 'Failed to delete bank account.';
                try {
                    const text = await response.text();
                    const json = JSON.parse(text);
                    if (json?.message) message = json.message;
                } catch {}
                await Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: message,
                });
            }
        } catch (err: any) {
            Swal.close();
            await Swal.fire({
                icon: 'error',
                title: 'Unexpected Error',
                text: err?.message ?? 'Something went wrong while deleting the account.',
            });
        }
    };

    const handleFormSuccess = () => {
        setShowForm(false);
        setEditingAccount(null);
        fetchBankAccounts();
        onSuccess?.()
    };

    const handleEdit = (account: BankAccount) => {
        setEditingAccount(account);
        setShowForm(true);
    };

    const handleAddNew = () => {
        setEditingAccount(null);
        setShowForm(true);
    };

    const handleCancel = () => {
        setShowForm(false);
        setEditingAccount(null);
    };

    const defaultTrigger = (
        <button
            className="cursor-pointer w-full flex items-center gap-4 px-4 py-2 bg-foreground/20 hover:bg-foreground/30 rounded-lg transition-colors text-left text-primary-foreground"
            type="button"
        >
            <Landmark className="w-4 h-4 text-white" />
            <div className="font-medium">Akaun Bank</div>
        </button>
    );

    return (
        <Sheet open={showModal} onOpenChange={setShowModal} modal={false}>
            <SheetTrigger asChild>
                {trigger ?? defaultTrigger}
            </SheetTrigger>

            <SheetContent
                side={side}
                className={`max-w-sm mx-auto rounded-t-2xl overflow-y-auto transition-all ${contentClassName}`}
            >
                <SheetHeader>
                    <SheetTitle className="flex items-center gap-2 text-white">
                        <Landmark className="w-5 h-5" />
                        Bank Accounts
                    </SheetTitle>
                    <SheetDescription className="text-background">
                        Manage your linked bank accounts
                    </SheetDescription>
                </SheetHeader>

                <div className="space-y-4 px-4 pb-4">
                    {showForm ? (
                        <BankAccountForm
                            bankAccount={editingAccount || undefined}
                            onSuccess={handleFormSuccess}
                            onCancel={handleCancel}
                        />
                    ) : (
                        <>
                            {/* Add Account Button */}
                            <div className="flex items-center justify-end">
                                <Button
                                    size="sm"
                                    onClick={handleAddNew}
                                    className="bg-primary-foreground hover:bg-primary-foreground/80 text-primary"
                                >
                                    <Plus className="w-4 h-4" />
                                    Add Account
                                </Button>
                            </div>

                            {/* Bank Accounts List */}
                            {isLoading ? (
                                <div className="flex items-center justify-center py-8">
                                    <Loader2 className="w-6 h-6 animate-spin text-primary-foreground" />
                                </div>
                            ) : bankAccounts.length === 0 ? (
                                <div className="text-center py-8 text-muted">
                                    <Landmark className="w-12 h-12 mx-auto mb-2 opacity-50" />
                                    <p>No bank accounts found</p>
                                    <p className="text-sm">Add your first bank account to get started</p>
                                </div>
                            ) : (
                                <div className="space-y-3">
                                    {bankAccounts.map((account) => (
                                        <div
                                            key={account.id}
                                            className="flex items-center gap-4 p-4 border border-border rounded-lg bg-foreground/10"
                                        >
                                            <div className="h-12 w-12 flex items-center justify-center rounded-md bg-primary-foreground">
                                                {account.bank?.logo ? (
                                                    <img
                                                        src={`/storage/${account.bank.logo}`}
                                                        alt={account.bank.name}
                                                        className="w-8 h-8 rounded"
                                                    />
                                                ) : (
                                                    <Landmark className="w-6 h-6 text-background" />
                                                )}
                                            </div>
                                            <div className="flex-1">
                                                <div className="font-medium text-white">{account.bank?.name}</div>
                                                <div className="text-sm text-muted-foreground">{account.account_number}</div>
                                                <div className="text-xs text-muted-foreground">{account.account_name}</div>
                                            </div>
                                            <div className="flex flex-col gap-2">
                                                <Badge
                                                    variant={account.is_active ? 'default' : 'secondary'}
                                                    className="text-xs"
                                                >
                                                    {account.is_active ? 'Active' : 'Inactive'}
                                                </Badge>
                                                <div className="flex gap-1">
                                                    <Button
                                                        size="sm"
                                                        variant="ghost"
                                                        onClick={() => handleToggleStatus(account)}
                                                        className="h-6 w-6 p-0 text-muted-foreground hover:text-white"
                                                    >
                                                        {account.is_active ? (
                                                            <EyeOff className="w-3 h-3" />
                                                        ) : (
                                                            <Eye className="w-3 h-3" />
                                                        )}
                                                    </Button>
                                                    <Button
                                                        size="sm"
                                                        variant="ghost"
                                                        onClick={() => handleEdit(account)}
                                                        className="h-6 w-6 p-0 text-muted-foreground hover:text-white"
                                                    >
                                                        <Edit className="w-3 h-3" />
                                                    </Button>
                                                    <Button
                                                        size="sm"
                                                        variant="ghost"
                                                        onClick={() => handleDelete(account)}
                                                        className="h-6 w-6 p-0 text-muted-foreground hover:text-red-400"
                                                    >
                                                        <Trash2 className="w-3 h-3" />
                                                    </Button>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            )}
                        </>
                    )}
                </div>
            </SheetContent>
        </Sheet>
    );
};

export default AccountBank;
