import { ChevronLeft, Globe } from "lucide-react";
import { useEffect, useRef, useState } from "react";
import { NewsProps } from "@/types";
import { useApiClient } from "@/lib/api";

const News = () => {
    const [currentSlide, setCurrentSlide] = useState(0);
    const [news, setNews] = useState<NewsProps[]>([]);
    const apiClient = useApiClient();

    const wrapperRef = useRef<HTMLDivElement | null>(null);
    const [slideW, setSlideW] = useState(0);

    // Measure container width (handles mobile DPR/orientation changes)
    useEffect(() => {
        if (!wrapperRef.current) return;
        const el = wrapperRef.current;
        const update = () => setSlideW(el.clientWidth);
        update();
        const ro = new ResizeObserver(update);
        ro.observe(el);
        window.addEventListener("orientationchange", update);
        return () => {
            ro.disconnect();
            window.removeEventListener("orientationchange", update);
        };
    }, []);

    useEffect(() => {
        let isMounted = true;
        (async () => {
            try {
                const response = await apiClient.get("/api/news");
                if (response.ok) {
                    const data: NewsProps[] = await response.json();
                    if (isMounted) {
                        setNews(Array.isArray(data) ? data : []);
                        setCurrentSlide(0);
                    }
                }
            } catch (e) {
                console.error("Error fetching news:", e);
            }
        })();
        return () => {
            isMounted = false;
        };
    }, [apiClient]);

    const total = news.length;

    useEffect(() => {
        if (total < 2) return;
        const id = setInterval(() => {
            setCurrentSlide((p) => (p + 1) % total);
        }, 3000);
        return () => clearInterval(id);
    }, [total]);

    useEffect(() => {
        if (total > 0 && currentSlide >= total) setCurrentSlide(0);
    }, [total, currentSlide]);

    const goPrev = () => total > 1 && setCurrentSlide((s) => (s - 1 + total) % total);
    const goNext = () => total > 1 && setCurrentSlide((s) => (s + 1) % total);

    // Use pixel translate for perfect alignment on mobile Safari
    const translatePx = currentSlide * slideW;

    return (
        <div className="space-y-2">
            <div className="flex items-center gap-2">
                <Globe className="w-5 h-5 text-white" />
                <h3 className="text-white text-lg font-semibold">NEWS BNM</h3>
            </div>

            <div ref={wrapperRef} className="relative rounded-lg overflow-hidden">
                <div
                    className="flex transition-transform duration-500 ease-in-out will-change-transform"
                    style={{ transform: `translate3d(-${translatePx}px,0,0)` }}
                >
                    {news.map((slide, index) => (
                        <div key={slide.id ?? index} className="w-full flex-none">
                            {/* Give the slide a real height: pick one */}
                            {/* Option A: aspect ratio */}
                            <div className="relative aspect-[16/9]">
                                {/* bg image layer */}
                                <div
                                    className="absolute inset-0 bg-primary-foreground"
                                    style={{
                                        backgroundImage: `url('/storage/${slide.image}')`,
                                        backgroundRepeat: "no-repeat",
                                        backgroundPosition: "center",
                                        backgroundSize: "cover",
                                    }}
                                />
                                {/* content overlay */}
                                <div className="absolute inset-0 flex items-center justify-center">
                                    <div className="bg-primary/30 rounded-md mx-2 backdrop-blur-sm text-center px-4 py-2 text-white">
                                        <div className="text-lg font-bold mb-1">{slide.title}</div>
                                        <div className="text-xs opacity-80 mb-1">{slide.description}</div>
                                    </div>
                                </div>
                            </div>

                            {/* Option B (if you prefer fixed heights):
                  <div className="relative h-40 sm:h-48 md:h-56 lg:h-64"> ...same children... </div>
              */}
                        </div>
                    ))}

                    {total === 0 && (
                        <div className="w-full flex-none">
                            <div className="relative h-40 sm:h-48 flex items-center justify-center text-white/70">
                                No news available.
                            </div>
                        </div>
                    )}
                </div>

                {total > 1 && (
                    <div className="absolute bottom-2 left-1/2 -translate-x-1/2 flex gap-2">
                        {news.map((_, index) => (
                            <button
                                key={index}
                                onClick={() => setCurrentSlide(index)}
                                className={`w-2 h-2 rounded-full transition-colors ${
                                    index === currentSlide ? "bg-white" : "bg-white/40"
                                }`}
                                aria-label={`Go to slide ${index + 1}`}
                            />
                        ))}
                    </div>
                )}

                <button
                    onClick={goPrev}
                    disabled={total < 2}
                    className="absolute left-2 top-1/2 -translate-y-1/2 w-8 h-8 bg-black/20 rounded-full flex items-center justify-center text-white hover:bg-black/40 transition-colors disabled:opacity-40 disabled:cursor-not-allowed"
                    aria-label="Previous slide"
                >
                    <ChevronLeft />
                </button>
                <button
                    onClick={goNext}
                    disabled={total < 2}
                    className="absolute right-2 top-1/2 -translate-y-1/2 w-8 h-8 bg-black/20 rounded-full flex items-center justify-center text-white hover:bg-black/40 transition-colors disabled:opacity-40 disabled:cursor-not-allowed"
                    aria-label="Next slide"
                >
                    <ChevronLeft className="rotate-180" />
                </button>
            </div>
        </div>
    );
};

export default News;
