import { PageProps } from '@/types';
import { CircleCheckBig, CircleX, Eye, EyeClosed, Mail, Power } from 'lucide-react';
import { useState, useEffect } from 'react';
import { router } from '@inertiajs/react';
import { cn, formatAmount } from '@/lib/utils';
import TicketSystem from '@/components/user/profile/ticket-system';
import SystemMessage from '@/components/user/system-message';
import { useApiClient } from '@/lib/api';
import Topup from '@/components/user/transactions/Topup';

const Header = ({ auth }: PageProps) => {
    const [showBalance, setShowBalance] = useState(false);
    const [unreadCount, setUnreadCount] = useState(0);
    const apiClient = useApiClient();

    // Fetch unread count on component mount
    useEffect(() => {
        const fetchData = async () => {
            try {
                // Fetch unread count
                const ticketsResponse = await apiClient.get(route('tickets.index'));

                if (ticketsResponse.ok) {
                    const ticketsData = await ticketsResponse.json();
                    const tickets = ticketsData.data || [];
                    let count = 0;

                    tickets.forEach((ticket: any) => {
                        if (!ticket.is_read) {
                            count++;
                        }
                    });

                    setUnreadCount(count);
                }
            } catch (error) {
                console.error('Failed to fetch data:', error);
            }
        };

        fetchData();

        // Refresh data every 30 seconds
        const interval = setInterval(fetchData, 30000);

        return () => clearInterval(interval);
    }, [apiClient]);

    return (
        <div className="relative bg-primary overflow-hidden">
            {/* Imej latar */}
            <div className="absolute inset-0 bg-[url('/background.jpg')] bg-no-repeat bg-top"></div>

            {/* Kandungan */}
            <div className="relative z-10 p-4 space-y-4">
                {/* Bar atas */}
                <div className="flex items-center justify-between gap-8">
                    {/* Logo bank */}
                    <div className="flex items-center px-8 py-1 h-12 rounded-full bg-white/30 backdrop-blur-sm p-1">
                        <img
                            src="/logo-full.svg"
                            alt="Logo Bank Negara Malaysia"
                            className="w-full h-full"
                        />
                    </div>

                    {/* Butang log keluar */}
                    <button
                        className="cursor-pointer w-10 h-10 bg-red-500 rounded-full flex items-center justify-center hover:bg-red-600 transition-colors"
                        onClick={() => router.post(route('logout'))}
                    >
                        <Power className="text-white " />
                    </button>
                </div>

                {/* Sapaan pengguna */}
                {/* Ticket Indicator */}
                <div
                    className="bg-white/50 backdrop-blur-sm rounded-lg px-4 py-2 flex flex-row items-center justify-between gap-4">
                    <div className="flex-1">
                        <div className="text-sm text-gray-600">Helo,</div>
                        <div className="font-medium text-gray-900 text-lg flex flex-row">
                            <span>{auth.user.name}</span>
                            {auth.user.is_verified ?
                                <div
                                    className="relative -top-2.5 -right-1 bg-w flex flex-row items-center space-x-0.5 text-[.5rem] text-green-600"
                                >
                                    <CircleCheckBig className="h-3 w-3" />
                                    <span className="whitespace-nowrap">PREMIUM</span>

                                </div> : <Topup trigger={<div
                                    className="cursor-pointer relative -top-2.5 -right-1 flex flex-row items-center space-x-0.5 text-[.5rem] text-destructive">
                                    <CircleX className="h-3 w-3" />
                                    <span className="whitespace-nowrap">VERIFIED NOW!</span>
                                </div>} />}
                            {/*<img*/}
                            {/*    src={auth.user.is_verified ? "/verified.png" : "/basic.png"}*/}
                            {/*    alt={auth.user.is_verified ? "Verified Member" : "Unverified Member"}*/}

                            {/*    title={auth.user.is_verified ? "Pengguna Disahkan" : "Pengguna Belum Disahkan"}*/}
                            {/*/>*/}
                        </div>
                    </div>
                    <TicketSystem
                        trigger={<button
                            className="cursor-pointer rounded-full w-8 h-8 bg-yellow-600 hover:bg-yellow-700 flex items-center justify-center relative">
                            {unreadCount > 0 && (
                                <span
                                    className="absolute -top-1 -right-1 text-xs text-white bg-red-500 rounded-full w-5 h-5 flex items-center justify-center font-medium">
                                {unreadCount > 9 ? '9+' : unreadCount}
                            </span>
                            )}
                            <Mail className="w-4 h-4 text-white" />
                        </button>}
                        onTicketOpen={() => setUnreadCount(0)}
                    />
                </div>

                {/* Kad baki */}
                <div
                    className="mt-4 bg-primary rounded-lg pl-3 px-2 py-2 text-white border-2 border-blue-900 flex flex-row items-center justify-between">
                    <div className="flex-1">
                        <div className="text-sm opacity-90">
                            <span>Baki Tersedia</span>
                        </div>
                        <div className="text-xl font-bold mt-1 flex flex-row items-center space-x-2">
                        <span>
                        {showBalance
                            ? auth.user.balance !== undefined
                                ? formatAmount(auth.user.balance)
                                : 'RM 0.00'
                            : '•••••••••'
                        }
                        </span>
                            <button
                                className="cursor-pointer transition-all"
                                onClick={() => setShowBalance(!showBalance)}
                            >
                                {showBalance && <Eye />}
                                {!showBalance && <EyeClosed />}
                            </button>
                        </div>
                    </div>
                    <SystemMessage />
                </div>
            </div>
        </div>
    );
};

export default Header;
