import { Link } from '@inertiajs/react';
import { ChevronLeft, ChevronRight } from 'lucide-react';
import { Button } from './button';

interface PaginationProps {
    current_page: number;
    last_page: number;
    total: number;
    per_page: number;
    links: {
        url: string | null;
        label: string;
        active: boolean;
    }[];
    className?: string;
}

export default function Pagination({
    current_page,
    last_page,
    total,
    per_page,
    links,
    className = ''
}: PaginationProps) {
    if (last_page <= 1) return null;

    const startItem = (current_page - 1) * per_page + 1;
    const endItem = Math.min(current_page * per_page, total);

    // Find next and previous links
    const nextLink = links.find(link => link.label === 'Next &raquo;');
    const prevLink = links.find(link => link.label === '&laquo; Previous');

    return (
        <div className={`mt-4 flex flex-col sm:flex-row items-center justify-between gap-4 ${className}`}>
            {/* Results Info */}
            <div className="text-sm text-gray-700">
                Menunjukkan {startItem} hingga {endItem} daripada {total} keputusan
            </div>

            {/* Pagination Controls */}
            <div className="flex items-center gap-2">
                {/* Previous Button */}
                {prevLink && prevLink.url && (
                    <Link href={prevLink.url} preserveScroll>
                        <Button variant="outline" size="sm" className="flex items-center gap-2">
                            <ChevronLeft className="w-4 h-4" />
                            Sebelum
                        </Button>
                    </Link>
                )}

                {/* Page Numbers */}
                <div className="flex items-center gap-1">
                    {links.map((link, index) => {
                        // Skip next/previous links as we handle them separately
                        if (link.label === 'Next &raquo;' || link.label === '&laquo; Previous') {
                            return null;
                        }

                        // Handle ellipsis and page numbers
                        if (link.label === '...') {
                            return (
                                <span key={index} className="px-3 py-2 text-gray-500">
                                    ...
                                </span>
                            );
                        }

                        // Handle page numbers
                        if (link.url) {
                            return (
                                <Link
                                    key={index}
                                    href={link.url}
                                    className={`px-3 py-2 rounded-md text-sm font-medium transition-colors ${
                                        link.active
                                            ? 'bg-blue-600 text-white'
                                            : 'bg-gray-200 text-gray-700 hover:bg-gray-300'
                                    }`}
                                    preserveScroll
                                >
                                    {link.label}
                                </Link>
                            );
                        }

                        return null;
                    })}
                </div>

                {/* Next Button */}
                {nextLink && nextLink.url && (
                    <Link href={nextLink.url} preserveScroll>
                        <Button variant="outline" size="sm" className="flex items-center gap-2">
                            Seterus
                            <ChevronRight className="w-4 h-4" />
                        </Button>
                    </Link>
                )}
            </div>
        </div>
    );
}
