import * as React from 'react'
import { X, ImagePlus } from 'lucide-react'
import { cn } from '@/lib/utils'

type CommonProps = {
    id?: string
    name?: string
    className?: string
    accept?: string
    disabled?: boolean
    error?: string
    /** Extra classes for the dropzone */
    dropzoneClassName?: string
}

/** Single-file mode (default if `multiple` is omitted) */
type SingleProps = CommonProps & {
    multiple?: false
    value?: File | null
    onChange?: (file: File | null) => void
    /** Existing server image to show while editing */
    existingUrl?: string | null
    /** Default preview (URL or File). Shown when value is empty. */
    defaultValue?: string | File | null
}

/** Multiple-file mode */
type MultiProps = CommonProps & {
    multiple: true
    value?: File[]
    onChange?: (files: File[]) => void
    /** Existing server images to show while editing */
    existingUrls?: string[]
    /** Default previews (URLs or Files). Shown before user picks files. */
    defaultValues?: Array<string | File>
}

type Props = SingleProps | MultiProps

const FileInput = React.forwardRef<HTMLInputElement, Props>((props, ref) => {
    const {
        id,
        name,
        className,
        accept = 'image/*',
        disabled,
        error,
        dropzoneClassName,
    } = props

    const inputRef = React.useRef<HTMLInputElement | null>(null)
    React.useImperativeHandle(ref, () => inputRef.current as HTMLInputElement)

    // Object URLs for *current* selected files (not server/default URLs)
    const [objectUrls, setObjectUrls] = React.useState<string[]>([])

    // Object URLs created from defaultValue(s) when they are File objects
    const defaultObjUrlsRef = React.useRef<string[]>([])
    const [defaultUrls, setDefaultUrls] = React.useState<string[]>([])

    // Revoke object URLs on change/unmount to avoid leaks
    React.useEffect(() => {
        return () => {
            objectUrls.forEach((u) => URL.revokeObjectURL(u))
            defaultObjUrlsRef.current.forEach((u) => URL.revokeObjectURL(u))
        }
    }, [objectUrls])

    // Keep previews in sync when parent-controlled `value` updates
    React.useEffect(() => {
        // Clear previous URLs
        setObjectUrls((prev) => {
            prev.forEach((u) => URL.revokeObjectURL(u))
            return []
        })

        if (props.multiple) {
            const files = props.value ?? []
            const urls = files.map((f) => URL.createObjectURL(f))
            setObjectUrls(urls)
        } else {
            const file = props.value ?? null
            const url = file ? URL.createObjectURL(file) : null
            setObjectUrls(url ? [url] : [])
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [props.multiple, (props as any).value])

    // Build defaultUrls from defaultValue(s) (strings or Files)
    React.useEffect(() => {
        // revoke previous default object urls
        defaultObjUrlsRef.current.forEach((u) => URL.revokeObjectURL(u))
        defaultObjUrlsRef.current = []

        const toUrl = (v: string | File): string => {
            if (typeof v === 'string') return v
            const u = URL.createObjectURL(v)
            defaultObjUrlsRef.current.push(u)
            return u
        }

        if (props.multiple) {
            const arr = props.defaultValues ?? []
            setDefaultUrls(arr.map(toUrl))
        } else {
            const v = props.defaultValue ?? null
            setDefaultUrls(v ? [toUrl(v)] : [])
        }
    }, [props.multiple, (props as any).defaultValues, (props as any).defaultValue])

    const handleFiles = (filesLike: FileList | File[] | null) => {
        const files = filesLike ? Array.from(filesLike as any as File[]) : []

        if (props.multiple) {
            props.onChange?.(files)
        } else {
            props.onChange?.(files[0] ?? null)
        }

        // reset input so picking the same file again retriggers onChange
        if (inputRef.current) inputRef.current.value = ''
    }

    const onDrop = (e: React.DragEvent) => {
        e.preventDefault()
        if (disabled) return
        handleFiles(e.dataTransfer.files)
    }

    // Build previews (defaults + existing first, then newly selected)
    type PreviewItem = { url: string; removable: boolean }
    let previews: PreviewItem[] = []

    if (props.multiple) {
        const existing = [
            ...(props.existingUrls ?? []),
            ...defaultUrls,
        ].map((u) => ({ url: u, removable: false }))

        const fresh = objectUrls.map((u) => ({ url: u, removable: true }))
        previews = [...existing, ...fresh]
    } else {
        const freshUrl = objectUrls[0]
        const fallback = defaultUrls[0] ?? props.existingUrl ?? null
        if (freshUrl) previews = [{ url: freshUrl, removable: true }]
        else if (fallback) previews = [{ url: fallback, removable: false }]
    }

    const existingCount = props.multiple
        ? ((props.existingUrls?.length ?? 0) + defaultUrls.length)
        : ((props.existingUrl || defaultUrls[0]) ? 1 : 0)

    const removeAt = (idx: number) => {
        if (!previews[idx]?.removable) return // only allow removing newly selected items
        if (props.multiple) {
            const current = (props.value ?? []).slice()
            // new selections start after existingCount
            const newIndex = idx - existingCount
            if (newIndex >= 0 && newIndex < current.length) {
                current.splice(newIndex, 1)
                props.onChange?.(current)
            }
        } else {
            props.onChange?.(null)
        }
    }

    return (
        <div className={cn('space-y-2', className)}>
            <div
                className={cn(
                    'rounded-lg border border-dashed p-4 text-sm',
                    'flex flex-col items-center justify-center gap-2',
                    'cursor-pointer select-none',
                    disabled && 'opacity-50 cursor-not-allowed',
                    dropzoneClassName
                )}
                onClick={() => !disabled && inputRef.current?.click()}
                onDragOver={(e) => e.preventDefault()}
                onDrop={onDrop}
                role="button"
                tabIndex={0}
                onKeyDown={(e) => {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault()
                        !disabled && inputRef.current?.click()
                    }
                }}
            >
                <ImagePlus className="h-6 w-6" />
                <div className="text-center">
                    <p className="font-medium">Click to upload</p>
                    <p className="text-muted-foreground">or drag &amp; drop</p>
                </div>

                <input
                    ref={inputRef}
                    id={id}
                    name={name}
                    type="file"
                    accept={accept}
                    hidden
                    multiple={Boolean((props as any).multiple)}
                    disabled={disabled}
                    onChange={(e) => handleFiles(e.currentTarget.files)}
                />
            </div>

            {!!previews.length && (
                <div className="grid grid-cols-3 gap-2">
                    {previews.map((p, idx) => (
                        <div key={idx} className="relative aspect-square overflow-hidden rounded-md border">
                            <img src={p.url} alt={`preview-${idx}`} className="h-full w-full object-cover" />
                            {p.removable && (
                                <button
                                    type="button"
                                    className="absolute top-1 right-1 inline-flex h-6 w-6 items-center justify-center rounded-full bg-white/80 shadow"
                                    onClick={(e) => {
                                        e.stopPropagation()
                                        removeAt(idx)
                                    }}
                                    aria-label="Remove image"
                                >
                                    <X className="h-4 w-4" />
                                </button>
                            )}
                        </div>
                    ))}
                </div>
            )}

            {error && <p className="text-sm text-red-500">{error}</p>}
        </div>
    )
})

FileInput.displayName = 'FileInput'
export default FileInput
