import { useState, useEffect } from 'react';
import { RefreshCw } from 'lucide-react';
import { Button } from './button';
import { Input } from './input';

interface CaptchaInputProps {
    name: string;
    id: string;
    placeholder?: string;
    required?: boolean;
    tabIndex?: number;
    className?: string;
    captchaValue?: string;
    onCaptchaChange?: (captcha: string) => void;
}

const CaptchaInput = ({
    name,
    id,
    placeholder = "Enter verification code",
    required = false,
    tabIndex,
    className = "",
    captchaValue = "",
    onCaptchaChange
}: CaptchaInputProps) => {
    const [captcha, setCaptcha] = useState<string>(captchaValue);
    const [inputValue, setInputValue] = useState<string>('');

    // Use server-provided captcha or generate new one
    useEffect(() => {
        if (captchaValue) {
            setCaptcha(captchaValue);
        } else {
            generateCaptcha();
        }
    }, [captchaValue]);

    // Generate random 6-digit number (fallback)
    const generateCaptcha = () => {
        const randomNum = Math.floor(100000 + Math.random() * 900000);
        const newCaptcha = randomNum.toString();
        setCaptcha(newCaptcha);
        if (onCaptchaChange) {
            onCaptchaChange(newCaptcha);
        }
    };

    const handleRefresh = () => {
        generateCaptcha();
        setInputValue(''); // Clear input when refreshing captcha
    };

    return (
        <div className={`relative ${className}`}>
            {/* Hidden input for captcha code */}
            <input
                type="hidden"
                name="captcha_code"
                value={captcha}
            />

            <Input
                id={id}
                name={name}
                type="number"
                placeholder={placeholder}
                required={required}
                tabIndex={tabIndex}
                value={inputValue}
                onChange={(e) => setInputValue(e.target.value)}
                className="rounded-full pr-32" // Add right padding for captcha
            />

            {/* Captcha suffix inside input */}
            <div className="absolute right-2 top-1/2 -translate-y-1/2 flex items-center gap-2">
                <div className="min-w-[60px] text-center">
                    <span className="text-sm font-mono font-bold text-gray-800 select-none">
                        {captcha}
                    </span>
                </div>

                {/* Refresh button */}
                <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={handleRefresh}
                    className="p-1 rounded-full border-gray-300 hover:bg-gray-50 h-6 w-6"
                    title="Refresh verification code"
                >
                    <RefreshCw className="w-3 h-3 text-gray-600" />
                </Button>
            </div>
        </div>
    );
};

export default CaptchaInput;
