import { Sheet, SheetContent, SheetDescription, SheetHeader, SheetTitle } from '@/components/ui/sheet';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Check, X, Clock, AlertCircle, CheckCircle, XCircle } from 'lucide-react';
import { cn, formatAmount, formatDate } from '@/lib/utils';
import {
    getStatusVariant,
    getTransactionStatusLabel
} from '@/components/user/transactions/TransactionItem';
import { router } from '@inertiajs/react';
import Swal from 'sweetalert2';
import { Transaction } from '@/types';

interface TransactionDetailSheetProps {
    isOpen: boolean;
    onClose: () => void;
    transaction: Transaction | null;
}

const getStatusIcon = (status: string) => {
    switch (status) {
        case 'completed':
            return <CheckCircle className="w-5 h-5 text-green-600" />;
        case 'processing':
            return <Clock className="w-5 h-5 text-yellow-600" />;
        case 'failed':
            return <XCircle className="w-5 h-5 text-red-600" />;
        case 'pending':
        default:
            return <AlertCircle className="w-5 h-5 text-yellow-600" />;
    }
};

export default function TransactionDetailSheet({ isOpen, onClose, transaction }: TransactionDetailSheetProps) {
    if (!transaction) return null;

    const handleApprove = async () => {
        const result = await Swal.fire({
            title: 'Sahkan Transaksi?',
            text: `Adakah anda pasti mahu meluluskan transaksi ${transaction.reference}?`,
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#10b981',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'Ya, Luluskan',
            cancelButtonText: 'Batal'
        });

        if (result.isConfirmed) {
            router.patch(route('admin.transactions.update-status', transaction.id), {
                status: 'completed'
            }, {
                onSuccess: () => {
                    Swal.fire({
                        title: 'Berjaya!',
                        text: 'Transaksi telah diluluskan',
                        icon: 'success',
                        timer: 2000
                    });
                    onClose();
                },
                onError: () => {
                    Swal.fire({
                        title: 'Ralat!',
                        text: 'Gagal meluluskan transaksi',
                        icon: 'error'
                    });
                }
            });
        }
    };

    const handleReject = async () => {
        const result = await Swal.fire({
            title: 'Tolak Transaksi?',
            text: `Adakah anda pasti mahu menolak transaksi ${transaction.reference}?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#ef4444',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'Ya, Tolak',
            cancelButtonText: 'Batal'
        });

        if (result.isConfirmed) {
            router.patch(route('admin.transactions.update-status', transaction.id), {
                status: 'failed'
            }, {
                onSuccess: () => {
                    Swal.fire({
                        title: 'Berjaya!',
                        text: 'Transaksi telah ditolak',
                        icon: 'success',
                        timer: 2000
                    });
                    onClose();
                },
                onError: () => {
                    Swal.fire({
                        title: 'Ralat!',
                        text: 'Gagal menolak transaksi',
                        icon: 'error'
                    });
                }
            });
        }
    };

    return (
        <Sheet open={isOpen} onOpenChange={onClose}>
            <SheetContent className="w-[600px] sm:w-[700px] overflow-y-auto" theme="light">
                <SheetHeader>
                    <SheetTitle className="flex items-center gap-2">
                        {getStatusIcon(transaction.status)}
                        <span>Butiran Transaksi</span>
                    </SheetTitle>
                    <SheetDescription>
                        Rujukan: {transaction.reference}
                    </SheetDescription>
                </SheetHeader>

                <div className="space-y-4 px-4">
                    {/* Transaction Status */}
                    <Card>
                        <CardHeader>
                            <CardTitle className="text-lg">Status Transaksi</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="flex items-center justify-between">
                                <Badge variant={getStatusVariant(transaction.status)}>
                                    {getTransactionStatusLabel(transaction.status)}
                                </Badge>
                                <span className="text-sm text-gray-600">
                                    {formatDate(transaction.updated_at)}
                                </span>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Transaction Details */}
                    <Card>
                        <CardHeader>
                            <CardTitle className="text-lg">Jumlah</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-4">
                            <div>
                                <p className={cn(
                                    'text-2xl font-semibold',
                                    transaction.amount > 0 ? 'text-green-600' : 'text-red-600'
                                )}>
                                    {formatAmount(transaction.amount)}
                                </p>
                            </div>
                        </CardContent>
                    </Card>

                    {/* User Information */}
                    <Card>
                        <CardHeader>
                            <CardTitle className="text-lg">Maklumat Pengguna</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-4">
                            <div>
                                <label className="text-sm font-medium text-gray-600">Nama</label>
                                <p className="text-sm text-gray-900">{transaction.user?.name}</p>
                            </div>
                            <div>
                                <label className="text-sm font-medium text-gray-600">Emel</label>
                                <p className="text-sm text-gray-900">{transaction.user?.email}</p>
                            </div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader>
                            <CardTitle className="text-lg">Nota</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-4">
                            <p className="text-sm text-gray-900">{transaction.notes}</p>
                        </CardContent>
                    </Card>

                    {/* Evidence */}
                    {transaction.evidence && (
                        <Card>
                            <CardHeader>
                                <CardTitle className="text-lg">Bukti Transaksi</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="aspect-video rounded-lg overflow-hidden border">
                                    <img
                                        src={`/storage/${transaction.evidence}`}
                                        alt="Bukti transaksi"
                                        className="w-full h-full object-cover"
                                    />
                                </div>
                            </CardContent>
                        </Card>
                    )}

                    {/* Timestamps */}
                    <Card>
                        <CardHeader>
                            <CardTitle className="text-lg">Maklumat Masa</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-2">
                            <div className="flex justify-between">
                                <span className="text-sm text-gray-600">Dicipta pada:</span>
                                <span className="text-sm text-gray-900">{formatDate(transaction.created_at)}</span>
                            </div>
                            <div className="flex justify-between">
                                <span className="text-sm text-gray-600">Dikemas kini pada:</span>
                                <span className="text-sm text-gray-900">{formatDate(transaction.updated_at)}</span>
                            </div>
                        </CardContent>
                    </Card>

                    {/* Action Buttons */}
                    {transaction.status === 'pending' && (
                        <Card>
                            <CardHeader>
                                <CardTitle className="text-lg">Tindakan</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="flex gap-3">
                                    <Button
                                        onClick={handleApprove}
                                        className="flex-1 bg-green-600 hover:bg-green-700 text-white"
                                    >
                                        <Check className="w-4 h-4 mr-2" />
                                        Luluskan
                                    </Button>
                                    <Button
                                        onClick={handleReject}
                                        variant="destructive"
                                        className="flex-1"
                                    >
                                        <X className="w-4 h-4 mr-2" />
                                        Tolak
                                    </Button>
                                </div>
                            </CardContent>
                        </Card>
                    )}
                </div>
            </SheetContent>
        </Sheet>
    );
}
