import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Plus, Minus, Wallet, Save, X } from 'lucide-react';
import { User } from '@/types';
import { toast } from 'react-toastify';
import { useApiClient } from '@/lib/api';
import { formatAmount } from '@/lib/utils';
import { usePage } from '@inertiajs/react';

interface BalanceManagementProps {
    user: User;
    isOpen: boolean;
    onClose: () => void;
    onBalanceUpdated: (newBalance: number) => void;
}

export default function BalanceManagement({ user, isOpen, onClose, onBalanceUpdated }: BalanceManagementProps) {
    const [action, setAction] = useState<'add' | 'reduce'>('add');
    const [amount, setAmount] = useState('');
    const [notes, setNotes] = useState('');
    const [isLoading, setIsLoading] = useState(false);
    const {csrf_token} = usePage().props;
    const apiClient = useApiClient();

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();

        if (!amount || parseFloat(amount) <= 0) {
            toast.error('Please enter a valid amount');
            return;
        }

        setIsLoading(true);

        try {
            const path = action === 'add'
                ? route('admin.users.add-balance', user.id)
                : route('admin.users.reduce-balance', user.id);

            const response = await apiClient.post(path, {
                body: { _token: csrf_token, amount: parseFloat(amount), notes },
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    toast.success(data.message);
                    onBalanceUpdated(data.new_balance);
                    handleClose();
                } else {
                    toast.error(data.message);
                }
            } else {
                toast.error('Failed to update balance');
            }
        } catch (error) {
            console.log(error)
            toast.error('Failed to update balance');
        } finally {
            setIsLoading(false);
        }
    };

    const handleClose = () => {
        setAction('add');
        setAmount('');
        setNotes('');
        onClose();
    };

    const currentBalance = user.balance || 0;

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/50">
            <Card className="w-full max-w-md mx-4">
                <CardHeader>
                    <CardTitle className="flex items-center justify-between">
                        <span className="flex items-center gap-2">
                            <Wallet className="w-5 h-5" />
                            Urus Baki Pengguna
                        </span>
                        <Button variant="ghost" size="sm" onClick={handleClose}>
                            <X className="w-4 h-4" />
                        </Button>
                    </CardTitle>
                </CardHeader>
                <CardContent>
                    {/* User Info */}
                    <div className="mb-4 p-3 bg-gray-50 rounded-lg">
                        <p className="text-sm font-medium text-gray-900">{user.name}</p>
                        <p className="text-sm text-gray-600">{user.email}</p>
                        <p className="text-lg font-bold text-green-600">
                            Baki Semasa: {formatAmount(currentBalance)}
                        </p>
                    </div>

                    {/* Action Tabs */}
                    <div className="flex mb-4 bg-gray-100 rounded-lg p-1">
                        <button
                            type="button"
                            onClick={() => setAction('add')}
                            className={`flex-1 py-2 px-3 rounded-md text-sm font-medium transition-colors ${
                                action === 'add'
                                    ? 'bg-white text-green-600 shadow-sm'
                                    : 'text-gray-600 hover:text-gray-900'
                            }`}
                        >
                            <Plus className="w-4 h-4 inline mr-2" />
                            Tambah Baki
                        </button>
                        <button
                            type="button"
                            onClick={() => setAction('reduce')}
                            className={`flex-1 py-2 px-3 rounded-md text-sm font-medium transition-colors ${
                                action === 'reduce'
                                    ? 'bg-white text-red-600 shadow-sm'
                                    : 'text-gray-600 hover:text-gray-900'
                            }`}
                        >
                            <Minus className="w-4 h-4 inline mr-2" />
                            Kurangkan Baki
                        </button>
                    </div>

                    <form onSubmit={handleSubmit} className="space-y-4">
                        {/* Amount Input */}
                        <div>
                            <Label htmlFor="amount">
                                Jumlah {action === 'add' ? 'Tambah' : 'Kurangkan'} (RM)
                            </Label>
                            <div className="relative mt-1">
                                <span className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500 font-bold">
                                    RM
                                </span>
                                <Input
                                    id="amount"
                                    type="number"
                                    value={amount}
                                    onChange={(e) => setAmount(e.target.value)}
                                    placeholder="0.00"
                                    min="0.01"
                                    step="0.01"
                                    className="pl-12"
                                    required
                                />
                            </div>
                        </div>

                        {/* Notes Input */}
                        <div>
                            <Label htmlFor="notes">Nota (Pilihan)</Label>
                            <Textarea
                                id="notes"
                                value={notes}
                                onChange={(e) => setNotes(e.target.value)}
                                placeholder="Sebab untuk perubahan baki..."
                                rows={3}
                                maxLength={500}
                            />
                        </div>

                        {/* Submit Button */}
                        <div className="flex gap-2 pt-2">
                            <Button
                                type="submit"
                                disabled={isLoading || !amount || parseFloat(amount) <= 0}
                                className={`flex-1 text-white ${
                                    action === 'add'
                                        ? 'bg-green-600 hover:bg-green-700'
                                        : 'bg-red-600 hover:bg-red-700'
                                }`}
                            >
                                <Save className="w-4 h-4 mr-2" />
                                {isLoading ? 'Menyimpan...' : action === 'add' ? 'Tambah Baki' : 'Kurangkan Baki'}
                            </Button>
                            <Button type="button" variant="outline" onClick={handleClose}>
                                Batal
                            </Button>
                        </div>
                    </form>
                </CardContent>
            </Card>
        </div>
    );
}
