import { useState, useEffect } from 'react';
import { useApiClient } from '@/lib/api';
import ChangePin from '@/components/user/profile/change-pin';
import { Auth } from '@/types';

interface PinCheckProps {
    children: React.ReactNode;
    auth: Auth;
}

const PinCheck = ({ children, auth }: PinCheckProps) => {
    const [hasPin, setHasPin] = useState<boolean | null>(null);
    const [isCheckingPin, setIsCheckingPin] = useState(true);
    const [showPinModal, setShowPinModal] = useState(false);
    const apiClient = useApiClient();

    useEffect(() => {
        const checkPinStatus = async () => {
            try {
                const response = await apiClient.get('/api/pin/check');
                if (response.ok) {
                    const data = await response.json();
                    setHasPin(data.has_pin);

                    // If user doesn't have PIN, show PIN creation modal
                    if (!data.has_pin) {
                        setShowPinModal(true);
                    }
                }
            } catch (error) {
                console.error('Error checking PIN status:', error);
                // If there's an error, assume user has PIN to avoid blocking
                setHasPin(true);
            } finally {
                setIsCheckingPin(false);
            }
        };

        checkPinStatus();
    }, [apiClient]);

    const handlePinSet = () => {
        setHasPin(true);
        setShowPinModal(false);
    };

    // Show loading while checking PIN status
    if (isCheckingPin) return null;

    // Show PIN creation modal if user doesn't have PIN
    if (!hasPin && showPinModal) {
        return (
            <div className="min-h-screen bg-primary">
                <div className="p-4">
                    <div className="text-center mb-8">
                        <h1 className="text-2xl font-bold text-white mb-2">
                            Selamat Datang!
                        </h1>
                        <p className="text-white/80">
                            Untuk keselamatan akaun anda, sila buat PIN transaksi terlebih dahulu.
                        </p>
                    </div>

                    <ChangePin
                        user={auth.user}
                        hasPin={false}
                        forceOpen={true}
                        onPinSet={handlePinSet}
                    />
                </div>
            </div>
        );
    }

    // Show main content if user has PIN
    return <>{children}</>;
};

export default PinCheck;
