import { ReactNode, useState } from 'react';
import { Link } from '@inertiajs/react';

interface DropdownProps {
    align?: 'left' | 'right';
    width?: '48' | string;
    contentClasses?: string;
    trigger: ReactNode;
    children: ReactNode;
}

export default function Dropdown({ align = 'right', width = '48', contentClasses = 'py-1 bg-white', trigger, children }: DropdownProps) {
    const [open, setOpen] = useState(false);

    const widthClass = {
        '48': 'w-48',
    }[width.toString()] || width;

    const alignmentClasses = (() => {
        if (align === 'left') {
            return 'origin-top-left left-0';
        } else if (align === 'right') {
            return 'origin-top-right right-0';
        }
        return 'origin-top';
    })();

    return (
        <div className="relative">
            <div onClick={() => setOpen(!open)}>
                {trigger}
            </div>

            <div
                className={`absolute z-50 mt-2 ${widthClass} rounded-md shadow-lg ${alignmentClasses}`}
                style={{
                    display: open ? 'block' : 'none',
                    transition: 'opacity 75ms ease-in-out, transform 100ms ease-in-out',
                    opacity: open ? 1 : 0,
                    transform: open ? 'scale(1)' : 'scale(0.95)',
                }}
                onClick={() => setOpen(false)}
            >
                <div className={`rounded-md ring-1 ring-black ring-opacity-5 ${contentClasses}`}>
                    {children}
                </div>
            </div>
        </div>
    );
}

Dropdown.Link = function DropdownLink({ className = '', children, ...props }: any) {
    return (
        <Link
            {...props}
            className={`block w-full px-4 py-2 text-left text-sm leading-5 text-gray-700 hover:bg-gray-100 focus:outline-none focus:bg-gray-100 transition duration-150 ease-in-out ${className}`}
        >
            {children}
        </Link>
    );
};

Dropdown.Button = function DropdownButton({ className = '', ...props }: any) {
    return (
        <button
            className={`block w-full px-4 py-2 text-left text-sm leading-5 text-gray-700 hover:bg-gray-100 focus:outline-none focus:bg-gray-100 transition duration-150 ease-in-out ${className}`}
            {...props}
        />
    );
};

Dropdown.Content = function DropdownContent({ children, className = '', ...props }: any) {
    return (
        <div className={className} {...props}>
            {children}
        </div>
    );
};

Dropdown.Trigger = function DropdownTrigger({ children, ...props }: any) {
    return (
        <div {...props}>
            {children}
        </div>
    );
};
