<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class SystemSetting extends Model
{
    protected $fillable = [
        'key',
        'value',
        'type',
        'description',
    ];

    /**
     * Get a setting value by key
     */
    public static function getValue(string $key, $default = null)
    {
        $setting = static::where('key', $key)->first();
        
        if (!$setting) {
            return $default;
        }

        return match($setting->type) {
            'integer' => (int) $setting->value,
            'boolean' => (bool) $setting->value,
            'json' => json_decode($setting->value, true),
            default => $setting->value,
        };
    }

    /**
     * Set a setting value by key
     */
    public static function setValue(string $key, $value, string $type = 'string', string $description = null): void
    {
        $setting = static::firstOrNew(['key' => $key]);
        
        $setting->value = match($type) {
            'json' => json_encode($value),
            default => (string) $value,
        };
        
        $setting->type = $type;
        $setting->description = $description;
        $setting->save();
    }

    /**
     * Get max upload size in bytes
     */
    public static function getMaxUploadSize(): int
    {
        return static::getValue('max_upload_size', 2048); // Default 2MB
    }

    /**
     * Get new user balance
     */
    public static function getNewUserBalance(): float
    {
        return static::getValue('new_user_balance', 0.00); // Default 0
    }
}
