<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SystemMessage extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'message',
        'is_active',
        'display_order'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'display_order' => 'integer',
    ];

    /**
     * Get active system messages ordered by display order
     */
    public static function getActiveMessages()
    {
        return static::where('is_active', true)
            ->orderBy('display_order')
            ->get();
    }

    /**
     * Get active system messages for a specific user
     */
    public static function getActiveMessagesForUser($userId)
    {
        return static::where('is_active', true)
            ->whereHas('users', function ($query) use ($userId) {
                $query->where('user_id', $userId);
            })
            ->orderBy('display_order')
            ->get();
    }

    /**
     * Scope for active messages
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for ordered messages
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('display_order');
    }

    /**
     * Users that can view this message
     */
    public function users()
    {
        return $this->belongsToMany(User::class, 'system_message_users');
    }
}
