<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\JsonResponse;

class PinController extends Controller
{
    /**
     * Check if user has PIN set
     */
    public function checkPin(): JsonResponse
    {
        $user = auth()->user();
        
        return response()->json([
            'has_pin' => !empty($user->pin_hash),
            'user_id' => $user->id
        ]);
    }

    /**
     * Create new PIN for user
     */
    public function createPin(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'pin' => 'required|string|size:6|regex:/^[0-9]+$/',
            'confirm_pin' => 'required|same:pin'
        ], [
            'pin.required' => 'PIN diperlukan',
            'pin.size' => 'PIN mesti 6 digit',
            'pin.regex' => 'PIN mesti nombor sahaja',
            'confirm_pin.same' => 'Pengesahan PIN tidak sepadan'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = auth()->user();
        
        // Check if user already has PIN
        if (!empty($user->pin_hash)) {
            return response()->json([
                'success' => false,
                'message' => 'User already has PIN set'
            ], 400);
        }

        // Hash and save PIN
        $user->pin_hash = Hash::make($request->pin);
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'PIN berjaya dicipta'
        ]);
    }

    /**
     * Update existing PIN
     */
    public function updatePin(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'old_pin' => 'required|string|size:6|regex:/^[0-9]+$/',
            'new_pin' => 'required|string|size:6|regex:/^[0-9]+$/',
            'confirm_pin' => 'required|same:new_pin'
        ], [
            'old_pin.required' => 'PIN lama diperlukan',
            'old_pin.size' => 'PIN lama mesti 6 digit',
            'old_pin.regex' => 'PIN lama mesti nombor sahaja',
            'new_pin.required' => 'PIN baru diperlukan',
            'new_pin.size' => 'PIN baru mesti 6 digit',
            'new_pin.regex' => 'PIN baru mesti nombor sahaja',
            'confirm_pin.same' => 'Pengesahan PIN tidak sepadan'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = auth()->user();
        
        // Check if user has PIN set
        if (empty($user->pin_hash)) {
            return response()->json([
                'success' => false,
                'message' => 'User does not have PIN set'
            ], 400);
        }

        // Verify old PIN
        if (!Hash::check($request->old_pin, $user->pin_hash)) {
            return response()->json([
                'success' => false,
                'message' => 'PIN lama tidak betul'
            ], 400);
        }

        // Hash and save new PIN
        $user->pin_hash = Hash::make($request->new_pin);
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'PIN berjaya dikemaskini'
        ]);
    }

    /**
     * Verify PIN for transactions
     */
    public function verifyPin(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'pin' => 'required|string|size:6|regex:/^[0-9]+$/'
        ], [
            'pin.required' => 'PIN diperlukan',
            'pin.size' => 'PIN mesti 6 digit',
            'pin.regex' => 'PIN mesti nombor sahaja'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = auth()->user();
        
        // Check if user has PIN set
        if (empty($user->pin_hash)) {
            return response()->json([
                'success' => false,
                'message' => 'User does not have PIN set'
            ], 400);
        }

        // Verify PIN
        if (!Hash::check($request->pin, $user->pin_hash)) {
            return response()->json([
                'success' => false,
                'message' => 'PIN tidak betul'
            ], 400);
        }

        return response()->json([
            'success' => true,
            'message' => 'PIN verified successfully'
        ]);
    }
}
