<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Bank;
use App\Models\BankAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class BankAccountController extends Controller
{
    /**
     * Display a listing of the user's bank accounts.
     */
    public function index()
    {
        $user = auth()->user();
        $bankAccounts = $user->bankAccounts()->with('bank')->get();
        $banks = Bank::where('is_active', true)->get();

        return response()->json([
            'bank_accounts' => $bankAccounts,
            'banks' => $banks,
        ]);
    }

    /**
     * Store a newly created bank account.
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'bank_id' => 'required|integer|exists:banks,id',
                'account_number' => 'required|string|max:50',
                'account_name' => 'required|string|max:255'
            ]);

            $user = auth()->user();

            // Check if account number already exists for this user
            $existingAccount = $user->bankAccounts()
                ->where('account_number', $validated['account_number'])
                ->where('bank_id', $validated['bank_id'])
                ->first();

            if ($existingAccount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bank account already exists for this user',
                ], 400);
            }

            // Create bank account
            $bankAccount = $user->bankAccounts()->create([
                'bank_id' => $validated['bank_id'],
                'account_number' => $validated['account_number'],
                'account_name' => $validated['account_name'],
                'is_active' => true,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Bank account added successfully',
                'bank_account' => $bankAccount->load('bank'),
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to add bank account',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update the specified bank account.
     */
    public function update(Request $request, BankAccount $bankAccount)
    {
        try {
            // Check if user owns this bank account
            if ($bankAccount->user_id != auth()->id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to bank account',
                ], 403);
            }

            $validated = $request->validate([
                'bank_id' => 'required|integer|exists:banks,id',
                'account_number' => 'required|string|max:50',
                'account_name' => 'required|string|max:255',
                'notes' => 'nullable|string|max:1000',
                'is_active' => 'boolean',
            ]);

            // Check if account number already exists for this user (excluding current account)
            $existingAccount = auth()->user()->bankAccounts()
                ->where('account_number', $validated['account_number'])
                ->where('bank_id', $validated['bank_id'])
                ->where('id', '!=', $bankAccount->id)
                ->first();

            if ($existingAccount) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bank account already exists for this user',
                ], 400);
            }

            // Update bank account
            $bankAccount->update($validated);

            return response()->json([
                'success' => true,
                'message' => 'Bank account updated successfully',
                'bank_account' => $bankAccount->load('bank'),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update bank account',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Remove the specified bank account.
     */
    public function destroy(BankAccount $bankAccount)
    {
        try {
            // Check if user owns this bank account
            if ($bankAccount->user_id != auth()->id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to bank account',
                ], 403);
            }

            $bankAccount->delete();

            return response()->json([
                'success' => true,
                'message' => 'Bank account deleted successfully',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete bank account',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Toggle the active status of a bank account.
     */
    public function toggleStatus(BankAccount $bankAccount)
    {
        try {
            // Check if user owns this bank account
            if ($bankAccount->user_id != auth()->id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized access to bank account',
                ], 403);
            }

            $bankAccount->update([
                'is_active' => !$bankAccount->is_active,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Bank account status updated successfully',
                'bank_account' => $bankAccount->load('bank'),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update bank account status',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
