<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class UserController extends Controller
{
    public function index(Request $request)
    {
        $query = User::query();

        // Search functionality
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', "%{$request->search}%")
                  ->orWhere('email', 'like', "%{$request->search}%")
                  ->orWhere('phone_number', 'like', "%{$request->search}%");
            });
        }

        // Filter by role
        if (!empty($request->role) && $request->role !== 'all') {
            $query->where('role', $request->role);
        }

        $users = $query->withCount('transactions')
            ->latest()
            ->paginate(15)
            ->withQueryString();

        // Append balance to each user
        $users->getCollection()->transform(function ($user) {
            $user->balance = $user->balance;
            return $user;
        });

        return Inertia::render('admin/users/index', [
            'users' => $users,
            'filters' => $request->only(['search', 'role']),
        ]);
    }

    public function create()
    {
        return Inertia::render('admin/users/create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'phone_number' => 'required|string|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:admin,user',
        ]);

        User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone_number' => $validated['phone_number'],
            'password' => Hash::make($validated['password']),
            'role' => $validated['role'],
        ]);

        if ($request->wantsJson()) {
            return response()->json(['message' => 'User created successfully.']);
        }

        return redirect()->route('admin.users.index')
            ->with('message', 'User created successfully.');
    }

    public function show(User $user)
    {
        $user->load(['transactions' => function ($query) {
            $query->latest()->take(20);
        }]);

        return Inertia::render('admin/users/show', [
            'user' => $user,
        ]);
    }

    public function edit(User $user)
    {
        return Inertia::render('admin/users/edit', [
            'user' => $user,
        ]);
    }

    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'phone_number' => ['required', 'string', 'max:255', Rule::unique('users')->ignore($user->id)],
            'role' => 'required|in:admin,user',
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone_number' => $validated['phone_number'],
            'role' => $validated['role'],
        ]);

        if (!empty($validated['password'])) {
            $user->update(['password' => Hash::make($validated['password'])]);
        }

        if ($request->wantsJson()) {
            return response()->json(['message' => 'User updated successfully.']);
        }

        return redirect()->route('admin.users.index')
            ->with('message', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        // Prevent admin from deleting themselves
        if ($user->id === auth()->id()) {
            if (request()->wantsJson()) {
                return response()->json(['error' => 'You cannot delete your own account.'], 422);
            }
            return back()->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        if (request()->wantsJson()) {
            return response()->json(['message' => 'User deleted successfully.']);
        }

        return redirect()->route('admin.users.index')
            ->with('message', 'User deleted successfully.');
    }

    /**
     * Toggle user verification status
     */
    public function toggleVerification(User $user)
    {
        // Prevent admin from changing their own verification status
        if (auth()->id() === $user->id) {
            return back()->with('error', 'You cannot change your own verification status.');
        }

        $user->update([
            'is_verified' => !$user->is_verified
        ]);

        return back()->with('message', 'User verification status updated successfully.');
    }

    public function addBalance(Request $request, User $user)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01|max:1000000',
            'notes' => 'nullable|string|max:500',
        ]);

        // Create transaction
        $user->transactions()->create([
            'type' => 'Penambahan System',
            'amount' => $validated['amount'],
            'status' => 'completed',
            'reference' => Str::ulid(),
            'recipient' => $user->name,
            'evidence' => null,
            'notes' => $validated['notes'] ?? 'Balance added by admin',
            'bank_account_id' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Balance added successfully',
            'new_balance' => $user->fresh()->balance,
        ]);
    }

    public function reduceBalance(Request $request, User $user)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01|max:1000000',
            'notes' => 'nullable|string|max:500',
        ]);

        // Check if user has sufficient balance
        if ($user->balance < $validated['amount']) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient balance',
            ], 400);
        }

        // Create transaction
        $user->transactions()->create([
            'type' => 'Pengurangan System',
            'amount' => -1 * abs($validated['amount']),
            'status' => 'completed',
            'reference' => Str::ulid(),
            'recipient' => $user->name,
            'evidence' => null,
            'notes' => $validated['notes'] ?? 'Balance reduced by admin',
            'bank_account_id' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Balance reduced successfully',
            'new_balance' => $user->fresh()->balance,
        ]);
    }
}
