<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;

class TransactionController extends Controller
{
    public function index(Request $request)
    {
        $query = Transaction::with('user');

        // Search functionality
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('reference', 'like', "%{$request->search}%")
                  ->orWhereHas('user', function ($userQuery) use ($request) {
                      $userQuery->where('name', 'like', "%{$request->search}%")
                               ->orWhere('email', 'like', "%{$request->search}%");
                  });
            });
        }

        // Filter by type
        if ($request->type) {
            $query->where('type', $request->type);
        }

        // Filter by status
        if ($request->status) {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->date_from) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->date_to) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $transactions = $query->latest()
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('admin/transactions/index', [
            'transactions' => $transactions,
            'filters' => $request->only(['search', 'type', 'status', 'date_from', 'date_to']),
            'types' => ['Tambah Baki', 'Pemindahan', 'Bayar Bil', 'Pengeluaran', 'Penambahan System', 'Pengurangan System'],
            'statuses' => ['pending', 'processing', 'completed', 'failed'],
        ]);
    }

    public function create()
    {
        $users = User::select('id', 'name', 'email')->get();

        return Inertia::render('admin/transactions/create', [
            'users' => $users,
            'types' => ['Tambah Baki', 'Pemindahan', 'Bayar Bil', 'Pengeluaran', 'Penambahan System', 'Pengurangan System'],
            'statuses' => ['pending', 'processing', 'completed', 'failed'],
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'evidence' => 'nullable|url|max:500',
        ]);

        Transaction::create($validated);

        return redirect()->route('admin.transactions.index')
            ->with('message', 'Transaction created successfully.');
    }

    public function show(Transaction $transaction)
    {
        $transaction->load('user');

        return Inertia::render('admin/transactions/show', [
            'transaction' => $transaction,
        ]);
    }

    public function edit(Transaction $transaction)
    {
        $users = User::select('id', 'name', 'email')->get();

        return Inertia::render('admin/transactions/edit', [
            'transaction' => $transaction,
            'users' => $users,
            'types' => ['Tambah Baki', 'Pemindahan', 'Bayar Bil', 'Pengeluaran', 'Penambahan System', 'Pengurangan System'],
            'statuses' => ['pending', 'processing', 'completed', 'failed'],
        ]);
    }

    public function update(Request $request, Transaction $transaction)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'type' => 'required|in:Tambah Baki,Pemindahan,Bayar Bil,Pengeluaran,Penambahan System,Pengurangan System',
            'amount' => 'required|numeric|min:0.01',
            'status' => 'required|in:pending,processing,completed,failed',
            'reference' => 'required|string|max:255|unique:transactions,reference,' . $transaction->id,
            'recipient' => 'required|string|max:255',
            'evidence' => 'nullable|url|max:500',
            'notes' => 'nullable|string|max:1000',
        ]);

        $transaction->update($validated);

        return redirect()->route('admin.transactions.index')
            ->with('message', 'Transaction updated successfully.');
    }

    public function destroy(Transaction $transaction)
    {
        $transaction->delete();

        return redirect()->route('admin.transactions.index')
            ->with('message', 'Transaction deleted successfully.');
    }

    public function updateStatus(Request $request, Transaction $transaction)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,processing,completed,failed',
        ]);

        $transaction->update($validated);

        return back()->with('message', 'Transaction status updated successfully.');
    }
}
