<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Ticket;
use App\Models\TicketReply;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class TicketController extends Controller
{

    public function index(Request $request)
    {
        $query = Ticket::with(['user', 'replies']);

        // Search functionality
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', "%{$request->search}%")
                  ->orWhere('description', 'like', "%{$request->search}%");
            });
        }

        // Filter by status
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $tickets = $query->latest()
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('admin/tickets/index', [
            'tickets' => $tickets,
            'filters' => $request->only(['search', 'status']),
        ]);
    }

    public function show(Ticket $ticket)
    {
        $ticket->load(['replies.user', 'user']);
        
        // Mark ticket as viewed by admin
        $ticket->update(['last_viewed_by_admin' => now()]);

        return Inertia::render('admin/tickets/show', [
            'ticket' => $ticket,
        ]);
    }

    public function updateStatus(Request $request, Ticket $ticket)
    {
        $validated = $request->validate([
            'status' => 'required|in:open,in_progress,resolved,closed',
        ]);

        $ticket->status = $validated['status'];

        if ($validated['status'] === 'resolved' && !$ticket->resolved_at) {
            $ticket->resolved_at = now();
        } elseif ($validated['status'] !== 'resolved') {
            $ticket->resolved_at = null;
        }

        $ticket->save();

        return back()->with('message', 'Ticket status updated successfully.');
    }

    public function reply(Request $request, Ticket $ticket)
    {
        $validated = $request->validate([
            'content' => 'required|string',
            'attachments.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $reply = new TicketReply();
        $reply->ticket_id = $ticket->id;
        $reply->user_id = auth()->id();
        $reply->content = $validated['content'];

        // Handle attachments
        if ($request->hasFile('attachments')) {
            $attachments = [];
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('public/tickets', 'public');
                $attachments[] = [
                    'name' => $file->getClientOriginalName(),
                    'path' => $path,
                    'type' => 'image'
                ];
            }
            $reply->attachments = $attachments;
        }

        $reply->save();

        // Mark ticket as unread for user when admin replies
        $ticket->update(['is_read' => false]);

        // Update ticket status to in_progress if it was open
        if ($ticket->status === 'open') {
            $ticket->status = 'in_progress';
            $ticket->save();
        }

        return back()->with('message', 'Reply added successfully.');
    }

    public function destroy(Ticket $ticket)
    {
        // Delete attachments
        if ($ticket->attachments) {
            foreach ($ticket->attachments as $attachment) {
                Storage::delete(str_replace('storage/', 'public/', $attachment['path']));
            }
        }

        // Delete reply attachments
        foreach ($ticket->replies as $reply) {
            if ($reply->attachments) {
                foreach ($reply->attachments as $attachment) {
                    Storage::delete(str_replace('storage/', 'public/', $attachment['path']));
                }
            }
        }

        $ticket->replies()->delete();
        $ticket->delete();

        return redirect()->route('admin.tickets.index')
            ->with('message', 'Ticket deleted successfully.');
    }

    public function getUnreadCount()
    {
        $tickets = Ticket::with(['replies.user'])->get();
        $count = 0;
        
        foreach ($tickets as $ticket) {
            $isUnread = false;
            
            // Check if ticket has replies
            if ($ticket->replies && $ticket->replies->count() > 0) {
                $lastAdminReply = $ticket->replies
                    ->filter(function ($reply) {
                        return $reply->user && $reply->user->role === 'admin';
                    })
                    ->sortByDesc('created_at')
                    ->first();
                
                $lastUserReply = $ticket->replies
                    ->filter(function ($reply) {
                        return $reply->user && $reply->user->role !== 'admin';
                    })
                    ->sortByDesc('created_at')
                    ->first();
                
                // Check if ticket has unread messages based on last viewed time
                if ($lastUserReply) {
                    if ($lastAdminReply) {
                        // If admin has replied, check if user replied after admin's last reply
                        if ($lastUserReply->created_at > $lastAdminReply->created_at) {
                            $isUnread = true;
                        }
                    } else {
                        // If admin hasn't replied yet, check if user replied after ticket was last viewed
                        if ($ticket->last_viewed_by_admin) {
                            $isUnread = $lastUserReply->created_at > $ticket->last_viewed_by_admin;
                        } else {
                            // If never viewed, consider it unread
                            $isUnread = true;
                        }
                    }
                    
                    // If ticket is unread, also check if admin has viewed it since the last user reply
                    if ($isUnread && $ticket->last_viewed_by_admin) {
                        $isUnread = $lastUserReply->created_at > $ticket->last_viewed_by_admin;
                    }
                }
            } else {
                // New ticket with no replies - check if admin has viewed it
                if (!$ticket->last_viewed_by_admin) {
                    $isUnread = true;
                }
            }
            
            if ($isUnread) {
                $count++;
            }
        }
        
        return response()->json(['unread_count' => $count]);
    }
}
