<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\News;
use App\Models\SystemSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

use Inertia\Inertia;

class NewsController extends Controller
{
    public function index(Request $request)
    {
        $query = News::with('user');

        // Search functionality
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', "%{$request->search}%")
                  ->orWhere('description', 'like', "%{$request->search}%");
            });
        }

        $news = $query->latest()
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('admin/news/index', [
            'news' => $news,
            'filters' => $request->only(['search']),
        ]);
    }

    public function create()
    {
        return Inertia::render('admin/news/create');
    }

    public function store(Request $request)
    {
        $image_max_upload_size = SystemSetting::getMaxUploadSize();
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            'image' => "nullable|image|mimes:jpeg,png,jpg,gif,svg|max:$image_max_upload_size",
        ]);

        $news = new News();
        $news->title = $validated['title'];
        $news->description = $validated['description'];
        $news->user_id = auth()->id();

        if ($request->hasFile('image')) {
            $news->image = $request->file('image')->store('transactions/news', 'public');
        }

        $news->save();

        return redirect()->route('admin.news.index')
            ->with('message', 'Berita berjaya dicipta.');
    }

    public function show(News $news)
    {
        $news->load('user');

        return Inertia::render('admin/news/show', [
            'news' => $news,
        ]);
    }

    public function edit(News $news)
    {
        return Inertia::render('admin/news/edit', [
            'news' => $news,
        ]);
    }

    public function update(Request $request, News $news)
    {
        $image_max_upload_size = SystemSetting::getMaxUploadSize();
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string|max:1000',
            'image' => "nullable|image|mimes:jpeg,png,jpg,gif,svg|max:$image_max_upload_size",
        ]);

        $news->title = $validated['title'];
        $news->description = $validated['description'];

        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($news->image) {
                Storage::delete(str_replace('storage/', 'public/', $news->image));
            }

            $path = $request->file('image')->store('public/news', 'public');
            $news->image = $path;
        }

        $news->save();

        return redirect()->route('admin.news.index')
            ->with('message', 'Berita berjaya dikemas kini.');
    }

    public function destroy(News $news)
    {
        if ($news->image) {
            Storage::delete(str_replace('storage/', 'public/', $news->image));
        }

        $news->delete();

        return redirect()->route('admin.news.index')
            ->with('message', 'Berita berjaya dipadam.');
    }
}
