<?php

namespace App\Http\Controllers\Admin;

use App\Models\Bank;
use App\Models\SystemSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class BankController extends BaseController
{
    protected string $viewPath = 'admin.banks';
    protected string $routePrefix = 'admin.banks';
    protected string $modelName = 'Bank';

    public function __construct()
    {
        parent::__construct();
    }

    public function index()
    {
        $banks = Bank::orderBy('name')
            ->latest()
            ->select(['id', 'name', 'logo', 'is_active', 'created_at'])
            ->paginate(10)
            ->withQueryString()
            ->through(function ($bank) {
                return [
                    'id' => $bank->id,
                    'name' => $bank->name,
                    'logo' => $bank->logo ? asset('storage/' . $bank->logo) : null,
                    'is_active' => $bank->is_active,
                    'created_at' => $bank->created_at->format('Y-m-d H:i:s'),
                ];
            });

        return Inertia::render('admin/banks/index', [
            'banks' => $banks,
            'filters' => request()->only(['search'])
        ]);
    }

    public function create()
    {
        return Inertia::render('Admin/banks/Create');
    }

    public function store(Request $request)
    {
        $image_max_upload_size = SystemSetting::getMaxUploadSize();
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'logo' => "nullable|image|mimes:jpeg,png,jpg,gif,svg|max:$image_max_upload_size",
            'is_active' => 'boolean',
        ]);

        $data = [
            'name' => $validated['name'],
            'is_active' => $request->boolean('is_active', false),
        ];

        if ($request->hasFile('logo')) {
            $path = $request->file('logo')->store('banks', 'public');
            $data['logo'] = $path;
        }

        Bank::create($data);

        return redirect()->route('admin.banks.index')
            ->with('success', 'Bank created successfully.');
    }

    public function show($id)
    {
        $bank = Bank::findOrFail($id);

        return Inertia::render('Admin/banks/Show', [
            'bank' => [
                'id' => $bank->id,
                'name' => $bank->name,
                'logo' => $bank->logo ? asset('storage/' . $bank->logo) : null,
                'is_active' => $bank->is_active,
                'created_at' => $bank->created_at->format('Y-m-d H:i:s'),
                'updated_at' => $bank->updated_at->format('Y-m-d H:i:s'),
            ]
        ]);
    }

    public function edit($id)
    {
        $bank = Bank::findOrFail($id);

        return Inertia::render('Admin/banks/Edit', [
            'bank' => [
                'id' => $bank->id,
                'name' => $bank->name,
                'logo' => $bank->logo ? asset('storage/' . $bank->logo) : null,
                'is_active' => $bank->is_active,
            ]
        ]);
    }

    public function update(Request $request, $id)
    {
        $bank = Bank::findOrFail($id);

        $image_max_upload_size = SystemSetting::getMaxUploadSize();
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'logo' => "nullable|image|mimes:jpeg,png,jpg,gif,svg|max:$image_max_upload_size",
            'is_active' => 'boolean',
        ]);

        $bank->name = $validated['name'];
        $bank->is_active = $request->boolean('is_active', false);

        if ($request->hasFile('logo')) {
            // Delete old logo if exists
            if ($bank->logo) {
                Storage::disk('public')->delete($bank->logo);
            }

            $path = $request->file('logo')->store('banks', 'public');
            $bank->logo = $path;
        }

        $bank->save();

        return redirect()->route('admin.banks.index')
            ->with('success', 'Bank updated successfully.');
    }

    public function destroy($id)
    {
        $bank = Bank::findOrFail($id);

        // Delete logo if exists
        if ($bank->logo) {
            Storage::disk('public')->delete($bank->logo);
        }

        $bank->delete();

        return redirect()->route('admin.banks.index')
            ->with('success', 'Bank deleted successfully.');
    }
}
