<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bank;
use App\Models\BankAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class BankAccountController extends Controller
{
    /**
     * Display a listing of the bank accounts.
     */
    public function index(Request $request)
    {
        $perPage = $request->input('per_page', 10);
        $search = $request->input('search', '');

        $banks = Bank::orderBy('name')->get();

        $bankAccounts = BankAccount::with('bank')
            ->whereNull('user_id')
            ->where(function ($q) use ($search) {
                $q->where('account_number', 'like', "%{$search}%")
                    ->orWhere('account_name', 'like', "%{$search}%")
                    ->orWhereHas('bank', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    });
            })
            ->orderBy('created_at', 'desc')
            ->paginate($perPage)
            ->withQueryString();

        return Inertia::render('admin/bank-account/index', [
            'bankAccounts' => $bankAccounts,
            'banks' => $banks,
            'filters' => [
                'search' => $search,
                'per_page' => $perPage,
            ],
        ]);
    }

    /**
     * Show the form for creating a new bank account.
     */
    public function create()
    {
        $banks = Bank::active()->get(['id', 'name', 'code']);

        return Inertia::render('admin/bank-account/form', [
            'banks' => $banks,
        ]);
    }

    /**
     * Store a newly created bank account in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'bank_id' => 'required|exists:banks,id',
            'account_number' => 'required|string|max:50|unique:bank_accounts,account_number',
            'account_name' => 'required|string|max:255',
            'is_active' => 'boolean',
        ]);

        try {
            DB::beginTransaction();

            $bankAccount = BankAccount::create($validated);

            DB::commit();

            return redirect()
                ->route('admin.bank-accounts.index')
                ->with('success', 'Bank account created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to create bank account. ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified bank account.
     */
    public function edit(BankAccount $bankAccount)
    {
        $banks = Bank::active()->get(['id', 'name', 'code']);

        return Inertia::render('admin/bank-account/form', [
            'bankAccount' => $bankAccount->load('bank'),
            'banks' => $banks,
        ]);
    }

    /**
     * Update the specified bank account in storage.
     */
    public function update(Request $request, BankAccount $bankAccount)
    {
        $validated = $request->validate([
            'bank_id' => 'required|exists:banks,id',
            'account_number' => 'required|string|max:50|unique:bank_accounts,account_number,' . $bankAccount->id,
            'account_name' => 'required|string|max:255',
            'is_active' => 'boolean',
        ]);

        try {
            DB::beginTransaction();

            $bankAccount->update($validated);

            DB::commit();

            return redirect()
                ->route('admin.bank-accounts.index')
                ->with('success', 'Bank account updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to update bank account. ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified bank account from storage.
     */
    public function destroy(BankAccount $bankAccount)
    {
        try {
            DB::beginTransaction();

            $bankAccount->delete();

            DB::commit();

            return redirect()
                ->route('admin.bank-accounts.index')
                ->with('success', 'Bank account deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to delete bank account. ' . $e->getMessage());
        }
    }

    /**
     * Toggle the active status of the specified bank account.
     */
    public function toggleStatus(BankAccount $bankAccount)
    {
        try {
            $bankAccount->update([
                'is_active' => !$bankAccount->is_active
            ]);

            return response()->json([
                'success' => true,
                'is_active' => $bankAccount->fresh()->is_active
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update status.'
            ], 500);
        }
    }
}
